package ru.yandex.crypta.api.rest.resource.lab;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.security.PermitAll;
import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.inject.Provider;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.FormParam;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.MoreObjects;
import com.google.common.base.Strings;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import javafx.util.Pair;

import ru.yandex.crypta.audience.proto.TUserDataStats;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lab.CustomIdentifier;
import ru.yandex.crypta.lab.Identifier;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.proto.AccessLevel;
import ru.yandex.crypta.lab.proto.EHashingMethod;
import ru.yandex.crypta.lab.proto.ELabIdentifierType;
import ru.yandex.crypta.lab.proto.EMatchingScope;
import ru.yandex.crypta.lab.proto.ESampleViewState;
import ru.yandex.crypta.lab.proto.Sample;
import ru.yandex.crypta.lab.proto.Subsamples;
import ru.yandex.crypta.lab.proto.TMatchingOptions;
import ru.yandex.crypta.lab.proto.TSampleGroupID;
import ru.yandex.crypta.lab.proto.TSampleView;
import ru.yandex.crypta.lab.proto.TSampleViewOptions;
import ru.yandex.crypta.lab.proto.TSimpleSampleStats;
import ru.yandex.crypta.lab.proto.TSimpleSampleStatsWithInfo;
import ru.yandex.crypta.lab.tables.SamplesTable;
import ru.yandex.crypta.lib.schedulers.JobView;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.lang.StringUtils;

@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class SampleResource extends CommonLabResource {
    private final Provider<SecurityContext> securityContextProvider;

    @Inject
    public SampleResource(LabService lab, Provider<SecurityContext> securityContextProvider) {
        super(lab);
        this.securityContextProvider = securityContextProvider;
    }

    @GET
    @ApiOperation(value = "Retrieve user samples")
    public List<Sample> getSamples() {
        return lab().samples().getAll();
    }

    @GET
    @Path("{id}")
    @ApiOperation(value = "Retrieve sample")
    public Sample getSample(@PathParam("id") @NotNull String id) {
        return lab().samples().getSample(id);
    }

    @GET
    @Path("{id}/groups")
    @ApiOperation(value = "Retrieve sample groups")
    public List<String> getSampleGroups(@PathParam("id") @NotNull String id) {
        return lab().samples().getGroups(id);
    }

    @GET
    @Path("{id}/stats")
    @ApiOperation(value = "Retrieve sample stats")
    public Delayed<TSimpleSampleStats> getSampleStats(@PathParam("id") @NotNull String id,
            @QueryParam("groupId") String groupId,
            @QueryParam("baseSampleId") String baseSampleId,
            @QueryParam("baseGroupId") String baseGroupId)
    {
        TSampleGroupID target = createSampleGroupID(id, groupId);
        Optional<TSimpleSampleStats> stats;
        if (baseSampleId == null || baseSampleId.isEmpty()) {
            stats = lab().getStats(target);
        } else {
            TSampleGroupID base = createSampleGroupID(baseSampleId, baseGroupId);
            stats = lab().getStats(target, base);
        }
        return new Delayed<>(stats.orElse(null), stats.isPresent());
    }

    @GET
    @Path("{id}/stats_by_segment")
    @ApiOperation(value = "Retrieve sample stats by segment")
    public Delayed<TSimpleSampleStats> getSampleStatsBySegmentId(@PathParam("id") @NotNull String id,
            @QueryParam("groupId") String groupId,
            @QueryParam("segmentId") String segmentId)
    {
        TSampleGroupID target = createSampleGroupID(id, groupId);
        Optional<TSimpleSampleStats> stats = lab().getStatsByExportId(target, segmentId);
        return new Delayed<>(stats.orElse(null), stats.isPresent());
    }

    private TSampleGroupID createSampleGroupID(String id, String groupId) {
        return TSampleGroupID.newBuilder()
                .setSampleID(Strings.nullToEmpty(id))
                .setGroupID(Strings.nullToEmpty(groupId))
                .build();
    }

    @GET
    @Path("global/stats")
    @ApiOperation(value = "Retrieve global stats")
    @PermitAll
    public TUserDataStats getGlobalStats() {
        return lab().getGlobalStats();
    }

    @GET
    @Path("{id}/view")
    @ApiOperation(value = "Retrieve all sample views")
    public List<TSampleView> getSampleViews(@PathParam("id") @NotNull String id) {
        return lab().samples().getViews(id);
    }

    @GET
    @Path("{id}/view/{view_id}")
    @ApiOperation(value = "Retrieve sample view")
    public TSampleView getSampleView(
            @PathParam("id") @NotNull String id,
            @PathParam("view_id") @NotNull String viewId)
    {
        return lab().samples().getView(id, viewId);
    }

    @GET
    @Path("{id}/view/{view_id}/min_size")
    @ApiOperation(value = "Retrieve minimal allowed size")
    public long getSampleViewMinSize(
            @PathParam("id") @NotNull String id,
            @PathParam("view_id") @NotNull String viewId
    )
    {
        return lab().samples().getViewMinimalSize(id, viewId);
    }

    @DELETE
    @Path("{id}/view/{view_id}")
    @ApiOperation(value = "Delete sample view")
    public TSampleView deleteSampleView(
            @PathParam("id") @NotNull String id,
            @PathParam("view_id") @NotNull String viewId)
    {
        return lab().samples().deleteView(id, viewId);
    }

    @POST
    @Path("{id}/describe")
    @ApiOperation(value = "Enforces to describe the sample")
    public Sample describeSample(@PathParam("id") @NotNull String id) {
        return lab().samples().describe(id);
    }

    @POST
    @Path("{id}/past_describe")
    @ApiOperation(value = "Enforce dated sample description")
    public Sample pastDescribeSample(@PathParam("id") @NotNull String id) {
        return lab().samples().pastDescribe(id);
    }

    @POST
    @ApiOperation(value = "Create sample")
    public Sample createSample(
            @ApiParam("Path to the sample, e.g. //tmp/sample") @QueryParam("path") @NotNull String path,
            @ApiParam("Name of the sample") @QueryParam("name") String name,
            @ApiParam("Type of ID") @QueryParam("idType") @NotNull String idType,
            @ApiParam("Name of the field that contains ID") @QueryParam("idKey") @NotNull String idKey,
            @ApiParam("Name of the field that contains date") @QueryParam("dateKey") String dateKey,
            @ApiParam("Name of the field to group by") @QueryParam("groupingKey") String groupingKey,
            @ApiParam("TTL of tables (in seconds)") @QueryParam("ttl") Long ttl,
            @ApiParam("Access level that controls scope of the sample") @QueryParam("accessLevel") AccessLevel accessLevel,
            @ApiParam("Max groups count") @QueryParam("maxGroups_count") @DefaultValue("10") Integer maxGroupsCount
    )
    {
        String requiredName = Optional.ofNullable(name).orElse("Sample");
        Long optionalTtl = Optional.ofNullable(ttl).orElse(SamplesTable.DEFAULT_TTL);
        CustomIdentifier customIdentifier = getCustomIdentifier(idType, idKey);

        if (accessLevel == AccessLevel.PUBLIC && !securityContextProvider.get().isUserInRole(Roles.Lab.ADMIN)) {
            throw Exceptions.forbidden(
                    String.format(
                            "User %s does not have roles to create public samples",
                            securityContextProvider.get().getUserPrincipal().getName()
                    ),
                    "WRONG_ROLE"
            );
        }

        return lab().samples().createSample(
                YPath.simple(path), requiredName, customIdentifier, dateKey, groupingKey, optionalTtl,
                accessLevel, maxGroupsCount
        );
    }

    @PUT
    @Path("{id}/set_user_set_id")
    @ApiOperation(value = "Update sample")
    public Sample setUserSetId(
            @ApiParam("Sample id") @PathParam("id") String id,
            @ApiParam("User set id") @QueryParam("user_set_id") String userSetId
    ) {
        return lab().samples().setUserSetId(id, userSetId);
    }

    @POST
    @Path("{id}/special_view")
    @ApiOperation(value = "Create some special view of the sample")
    @Deprecated
    public TSampleView createSpecialView(
            @ApiParam("Sample ID") @PathParam("id") @NotNull String id,
            @ApiParam("Matching mode") @QueryParam("mode") @NotNull String mode
    )
    {
        return lab().samples().createCryptaIdStatisticsView(id);
    }

    @POST
    @Path("{id}/crypta_id_statistics_view")
    @ApiOperation(value = "Creates CryptaID statistics view of the sample")
    public TSampleView createCryptaIdStatisticsView(
            @ApiParam("SampleId") @PathParam("id") @NotNull String id
    ) {
        return lab().samples().createCryptaIdStatisticsView(id);
    }

    @POST
    @Path("{id}/lookalike_view")
    @ApiOperation(value = "Creates Lookalike view of the sample")
    public TSampleView createLookalikeView(
            @ApiParam("SampleId") @PathParam("id") @NotNull String id,
            @ApiParam("count of lal yuid's in output") @QueryParam("outputCount") @NotNull Integer outputCount,
            @ApiParam("whether to use dates associated with ids") @QueryParam("useDates") @NotNull Boolean useDates
    ) {
        return lab().samples().createLookalikeView(id, outputCount, useDates);
    }

    @Deprecated
    @POST
    @Path("{id}/view")
    @ApiOperation(value = "Match sample with some identifiers")
    public TSampleView createView(
            @ApiParam("Sample ID") @PathParam("id") @NotNull String id,
            @ApiParam("Method of identifiers hashing") @QueryParam("hashingMethod") @NotNull EHashingMethod hashingMethod,
            @ApiParam("Target identifier") @QueryParam("identifier") @NotNull ELabIdentifierType idType,
            @ApiParam("Whether to match with original identifiers") @QueryParam("includeOriginal") @NotNull Boolean includeOriginal,
            @ApiParam("Name of the field that contains ID") @QueryParam("key") String key,
            @ApiParam("Scope of matching (cross-device, in-device)") @QueryParam("scope") EMatchingScope scope
    )
    {
        return createMatchingView(id, hashingMethod, idType, includeOriginal, key, scope);
    }

    @POST
    @Path("{id}/matching_view")
    @ApiOperation(value = "Match sample with some identifiers")
    public TSampleView createMatchingView(
            @ApiParam("Sample ID") @PathParam("id") @NotNull String id,
            @ApiParam("Method of identifiers hashing") @QueryParam("hashingMethod") @NotNull EHashingMethod hashingMethod,
            @ApiParam("Target identifier") @QueryParam("identifier") @NotNull ELabIdentifierType idType,
            @ApiParam("Whether to match with original identifiers") @QueryParam("includeOriginal") @NotNull Boolean includeOriginal,
            @ApiParam("Name of the field that contains ID") @QueryParam("key") String key,
            @ApiParam("Scope of matching (cross-device, in-device)") @QueryParam("scope") EMatchingScope scope
    ) {
        TMatchingOptions.Builder matchingOptions = TMatchingOptions.newBuilder()
                .setHashingMethod(hashingMethod)
                .setIdType(idType)
                .setIncludeOriginal(includeOriginal)
                .setScope(MoreObjects.firstNonNull(scope, EMatchingScope.CROSS_DEVICE));
        if (Objects.nonNull(key)) {
            if (key.isEmpty()) {
                throw Exceptions.illegal("Key cannot be empty");
            }
            matchingOptions.setKey(key);
        }
        TSampleViewOptions.Builder options = TSampleViewOptions.newBuilder()
                .setMatching(matchingOptions);

        return lab().samples().createMatchingView(id, options.build());
    }

    @POST
    @Path("{id}/learn")
    @ApiOperation(value = "Run learning task with sample")
    public Sample runLearningTask(
            @ApiParam("Sample ID") @PathParam("id") @NotNull String sampleId
    ) {
        return lab().samples().learn(sampleId);
    }

    private CustomIdentifier getCustomIdentifier(String idType, String idKey) {
        if (Objects.nonNull(idType)) {
            Identifier usualIdentifier = Identifier.byName(idType);
            if (Objects.nonNull(idKey)) {
                return new CustomIdentifier(idKey, usualIdentifier);
            } else {
                return new CustomIdentifier(usualIdentifier.getName(), usualIdentifier);
            }
        } else {
            throw Exceptions.illegal("ID type can't be null");
        }
    }

    @PUT
    @Path("{id}/source")
    @ApiOperation(value = "Update sample")
    public Sample updateSampleSource(
            @PathParam("id") @NotNull String id,
            @QueryParam("path") @NotNull String path
    )
    {
        return lab().samples().updateSampleSource(id, path);
    }

    @PUT
    @Path("{id}/state")
    @ApiOperation(value = "Update sample state")
    @RolesAllowed({Roles.ADMIN})
    public Sample updateSampleState(
            @PathParam("id") @NotNull String id,
            @QueryParam("state") @NotNull String state
    )
    {
        return lab().samples().updateSampleState(id, state);
    }

    @POST
    @Path("upload_to_siberia")
    @ApiOperation(value = "Upload sample to Siberia")
    public String uploadSampleToSiberia(
            @ApiParam("YT table path") @QueryParam("path") @NotNull String path,
            @ApiParam("Siberia user set title") @QueryParam("title") @NotNull String title,
            @ApiParam("Type of ID") @QueryParam("idType") @NotNull String idType,
            @ApiParam("Name of the field that contains ID") @QueryParam("idKey") @NotNull String idKey,
            @ApiParam("Ttl in seconds") @QueryParam("ttl") @NotNull Long ttl
    )
    {
        CustomIdentifier customIdentifier = getCustomIdentifier(idType, idKey);
        return lab().samples().uploadSampleToSiberia(YPath.simple(path), title, customIdentifier, ttl);
    }

    @DELETE
    @Path("{id}")
    @ApiOperation(value = "Delete sample")
    public Sample deleteSample(@PathParam("id") @NotNull String id) {
        return lab().samples().deleteSample(id);
    }

    @GET
    @Path("{id}/jobs")
    @ApiOperation(value = "Get jobs of sample")
    public List<JobView> getSampleJobs(@PathParam("id") @NotNull String id) {
        return lab().samples().getSampleJobs(id);
    }

    @DELETE
    @Path("remove_outdated")
    @RolesAllowed({Roles.ADMIN})
    public List<TSampleView> deleteOutdated() {
        return lab().samples().deleteOutdatedSampleViews();
    }

    @PUT
    @Path("{id}/view/{view_id}/state")
    @RolesAllowed({Roles.ADMIN})
    public TSampleView updateSampleViewState(
            @PathParam("id") @NotNull String sampleId,
            @PathParam("view_id") @NotNull String viewId,
            @QueryParam("state") @NotNull ESampleViewState state)
    {
        return lab().samples().updateViewState(sampleId, viewId, state);
    }

    @PUT
    @Path("{id}/view/{view_id}/error")
    @ApiOperation(value = "Update sample view error")
    @RolesAllowed({Roles.ADMIN})
    public TSampleView updateSampleViewError(
            @PathParam("id") @NotNull String sampleId,
            @PathParam("view_id") @NotNull String viewId,
            @QueryParam("error") @NotNull String error)
    {
        return lab().samples().updateViewError(sampleId, viewId, error);
    }

    @GET
    @Path("read_view_from_yt")
    public List<JsonNode> getSampleViewFromYtByPages(
            @QueryParam("sample_id") @NotNull String sampleId,
            @QueryParam("view_id") @NotNull String viewId,
            @QueryParam("page") @NotNull Long page,
            @QueryParam("page_size") @DefaultValue("50") @NotNull Integer pageSize
    )
    {
        return lab().samples().getSamplePagesFromYt(sampleId, viewId, page, pageSize);
    }

    @GET
    @Path("read_from_siberia")
    public Response getSampleFromSiberia(
            @ApiParam("Siberia user set id") @QueryParam("user_set_id") @NotNull String userSetId,
            @ApiParam("Last user_id retrieved") @QueryParam("last_user_id") String lastUserId,
            @ApiParam("Limit") @QueryParam("limit") @DefaultValue("10") long limit
    )
    {
        var response = lab().samples().getSampleFromSiberia(userSetId, lastUserId, limit);
        return Response.status(response.getHttpCode()).entity(response.getText()).build();
    }

    @POST
    @Path("describe_by_siberia")
    @ApiOperation(value = "Upload sample to Siberia")
    public Response describeBySiberia(
            @ApiParam("Siberia user set id") @QueryParam("user_set_id") @NotNull String userSetId
    )
    {
        var response = lab().samples().describeBySiberia(userSetId);
        return Response.status(response.getHttpCode()).entity(response.getText()).build();
    }

    @GET
    @Path("get_stats_from_siberia")
    public TSimpleSampleStatsWithInfo getStatsFromSiberia(
            @ApiParam("Siberia user set id") @QueryParam("user_set_id") @NotNull String userSetId,
            @ApiParam("Base user set id") @QueryParam("base_user_set_id") String baseUserSetId
    )
    {
        if (baseUserSetId == null || baseUserSetId.isEmpty()) {
            var stats = lab().getStatsFromSiberia(userSetId);
            return stats.orElse(null);
        }

        var stats = lab().getStatsFromSiberia(new Pair<>(userSetId, baseUserSetId));
        return stats.orElse(null);

    }

    @POST
    @ApiOperation(value = "Create siberia segment")
    @Path("create_siberia_segment")
    public Response createSiberiaSegment(
            @ApiParam("Siberia user set id") @QueryParam("user_set_id") @NotNull String userSetId,
            @ApiParam("Title") @QueryParam("title") @NotNull String title,
            @ApiParam("Rule") @QueryParam("rule") @NotNull String rule
    )
    {
        var response = lab().samples().createSiberiaSegment(userSetId, title, rule);
        return Response.status(response.getHttpCode()).entity(response.getText()).build();
    }

    @POST
    @ApiOperation(value = "Remove siberia segment")
    @Path("remove_siberia_segment")
    public Response removeSiberiaSegment(
            @ApiParam("Siberia user set id") @QueryParam("user_set_id") @NotNull String userSetId,
            @ApiParam("Siberia segment id") @QueryParam("segment_id") @NotNull String segmentId
    )
    {
        var response = lab().samples().removeSiberiaSegment(userSetId, segmentId);
        return Response.status(response.getHttpCode()).entity(response.getText()).build();
    }

    @GET
    @Path("get_segments_from_siberia")
    @ApiOperation("Get segments from Siberia")
    public Response getSegmentsFromSiberia(
            @ApiParam("Siberia user set id") @QueryParam("user_set_id") @NotNull String userSetId,
            @ApiParam("Last segment id retrieved") @QueryParam("last_segment_id") String lastSegmentId,
            @ApiParam("Limit") @QueryParam("limit") @DefaultValue("10") long limit
    )
    {
        var response = lab().samples().getSegmentsFromSiberia(userSetId, lastSegmentId, limit);
        return Response.status(response.getHttpCode()).entity(response.getText()).build();
    }

    @GET
    @Path("get_siberia_segment_users")
    @ApiOperation("Get users from Siberia for given segment")
    public Response getSiberiaSegmentUsers(
            @ApiParam("Siberia user set id") @QueryParam("user_set_id") @NotNull String userSetId,
            @ApiParam("Siberia segment id") @QueryParam("segment_id") @NotNull String segmentId,
            @ApiParam("Last user id retrieved") @QueryParam("last_user_id") String lastUserId,
            @ApiParam("Limit") @QueryParam("limit") @DefaultValue("10") long limit
    )
    {
        var response = lab().samples().getSiberiaSegmentUsers(userSetId, segmentId, lastUserId, limit);
        return Response.status(response.getHttpCode()).entity(response.getText()).build();
    }

    @DELETE
    @Path("remove_siberia_user_set")
    @ApiOperation("Remove user set from Siberia")
    public Response removeSiberiaUserSet(
            @ApiParam("Siberia user set id") @QueryParam("user_set_id") @NotNull String userSetId
    )
    {
        var response = lab().samples().removeSiberiaUserSet(userSetId);
        return Response.status(response.getHttpCode()).entity(response.getText()).build();
    }

    public static class Delayed<T> {

        private final T value;
        private final boolean ready;

        public Delayed(T value, boolean ready) {
            this.value = value;
            this.ready = ready;
        }

        public T getValue() {
            return value;
        }

        public boolean isReady() {
            return ready;
        }

    }

    @GET
    @Path("{id}/subsamples")
    @ApiOperation("Get subsamples by sample id")
    public Subsamples getSubsamples(@PathParam("id") String sampleId) {
        return lab().samples().getSubsamples(sampleId);
    }

    @POST
    @Consumes(MediaType.APPLICATION_FORM_URLENCODED)
    @Path("{id}/subsamples")
    @ApiOperation("Creates map of user_set_id and grouping key value for sample id")
    public Subsamples createSubsamples(
            @ApiParam("Parent sample id") @PathParam("id") @NotNull String sampleId,
            @ApiParam("List of comma separated values <user_set_id,grouping_key_value>")
            @FormParam("ids") @NotNull List<String> ids
    ) {
        Map<String, String> userSetIdToGroupingKeyValue = new HashMap<>();
        for (String idsStr : ids) {
            String[] idPair = StringUtils.split(idsStr, ",");
            userSetIdToGroupingKeyValue.put(idPair[0], idPair[1]);
        }

        return lab().samples().createSubsamples(
                sampleId, userSetIdToGroupingKeyValue
        );
    }
}
