package ru.yandex.crypta.api.rest.resource.lab;

import java.util.List;
import java.util.stream.Collectors;

import javax.inject.Inject;
import javax.inject.Provider;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.SecurityContext;

import com.fasterxml.jackson.databind.JsonNode;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.clients.audience.AudienceClient;
import ru.yandex.crypta.clients.audience.AudienceCreateGrantResponse;
import ru.yandex.crypta.clients.audience.AudienceDeleteResponse;
import ru.yandex.crypta.clients.audience.AudienceListGrantsResponse;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.exception.NotAuthenticatedException;
import ru.yandex.crypta.common.exception.WrongRequestException;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.graph.Identifier;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lab.proto.AudienceCreateGrantsResponse;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.service.staff.StaffService;


@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class SegmentExportAudienceResource {
    private final AudienceClient client;
    private final StaffService staffService;
    private final Provider<SecurityContext> securityContextProvider;
    private static final Logger LOG = LoggerFactory.getLogger(LabResource.class);

    @Inject
    SegmentExportAudienceResource(AudienceClient client, StaffService staffService, Provider<SecurityContext> securityContextProvider) {
        this.client = client;
        this.staffService = staffService;
        this.securityContextProvider = securityContextProvider;
    }

    @GET
    @ApiOperation(value = "Check if login bound to Staff")
    @Path("isStaffLogin")
    public boolean isStaffLogin(
            @QueryParam("login") @NotNull String login
    ) {
        var identifier = new Identifier(EIdType.LOGIN, login).getNormalizedValue();
        var allBoundLogins = staffService.getAllBoundLogins();
        var isBoundToStaff = allBoundLogins.containsKey(identifier);

        if (isBoundToStaff) {
            LOG.info(String.format("Login %s is bound to %s Staff account", identifier, allBoundLogins.get(identifier)));
        }

        return isBoundToStaff;
    }

    @GET
    @Path("segments")
    @ApiOperation(value = "Lists accessible segments")
    public JsonNode listAudienceSegments() {
        return client.listSegments();
    }

    @GET
    @Path("{segmentId}")
    @ApiOperation(value = "Gets Audience's segment by id")
    public JsonNode getAudienceSegment(@PathParam("segmentId") @NotNull String segmentId) {
        return client.getSegment(segmentId);
    }

    @DELETE
    @Path("{segmentId}")
    @ApiOperation(value = "Deletes Audience's segment by id")
    public AudienceDeleteResponse deleteAudienceSegment(@PathParam("segmentId") @NotNull String segmentId) {
        return client.deleteSegment(segmentId);
    }

    @GET
    @Path("{segmentId}/grants")
    @ApiOperation(value = "Lists grants of segment")
    public AudienceListGrantsResponse listGrants(@PathParam("segmentId") @NotNull String segmentId) {
        return client.listGrants(segmentId);
    }

    @PUT
    @Path("{segmentId}/grants")
    @ApiOperation(value = "Sets grants to segment")
    public List<AudienceCreateGrantsResponse> createGrants(
            @PathParam("segmentId") @NotNull String segmentId,
            @QueryParam("logins") @NotNull List<String> logins,
            @QueryParam("comment") @DefaultValue("Shared from Laboratory") String comment
    )
    {
        var allowedToShareSegment =  isAllowedToShareSegment();
        var requestAuthor = securityContextProvider.get().getUserPrincipal().getName();

        return logins
            .stream()
            .parallel()
            .map(login -> processLogin(segmentId, login, comment, allowedToShareSegment, requestAuthor))
            .collect(Collectors.toList());
    }

    @DELETE
    @Path("{segmentId}/grants")
    @ApiOperation(value = "Deletes user's grant")
    public AudienceDeleteResponse deleteGrant(
            @PathParam("segmentId") @NotNull String segmentId,
            @QueryParam("userLogin") @NotNull String userLogin
    )
    {
        return client.deleteGrant(segmentId, userLogin);
    }

    @PUT
    @Path("{segmentId}/update")
    @ApiOperation(value = "Updates segments name")
    public JsonNode updateAudienceName(
            @PathParam("segmentId") @NotNull String segmentId,
            @QueryParam("name") @NotNull String name
    ) {
        return client.updateSegment(segmentId, name);
    }

    private AudienceCreateGrantResponse createSingleGrant(
            String segmentId,
            String userLogin,
            String comment,
            boolean allowedToShareSegment,
            String requestAuthor
    ) {
        if (allowedToShareSegment || isStaffLogin(userLogin)) {
            return client.createGrant(segmentId, userLogin, comment);
        }

        throw Exceptions.notAuthenticated(
            String.format(
                "Login %s is not bound to Staff. User %s does not have roles to create audience grants for external logins",
                userLogin,
                requestAuthor
            )
        );
    }

    private AudienceCreateGrantsResponse processLogin(
            String segmentId,
            String login,
            String comment,
            boolean allowedToShareSegment,
            String requestAuthor
    ) {
        var grant = AudienceCreateGrantsResponse.newBuilder().setRequestedLogin(login);
        try {
            grant.setReceivedLogin(createSingleGrant(segmentId, login, comment, allowedToShareSegment, requestAuthor).getGrant().getUser());
        } catch (NotAuthenticatedException | WrongRequestException e) {
            AudienceCreateGrantsResponse.Error.Reason status;

            if (e.getClass() == NotAuthenticatedException.class) {
                status = AudienceCreateGrantsResponse.Error.Reason.NOT_AUTHENTICATED;
            } else {
                status = AudienceCreateGrantsResponse.Error.Reason.WRONG_REQUEST;
            }

            var error = AudienceCreateGrantsResponse.Error.newBuilder()
                .setReason(status)
                .setMessage(e.getMessage())
                .build();

            grant.setError(error);
        }
        return grant.build();
    }

    private boolean isAllowedToShareSegment() {
        return  securityContextProvider.get().isUserInRole(Roles.Lab.AUDIENCE_SEGMENTS_SHARER) ||
                securityContextProvider.get().isUserInRole(Roles.Lab.ADMIN);
    }
}
