package ru.yandex.crypta.api.rest.resource.lab;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.inject.Provider;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.FormParam;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.SecurityContext;

import io.swagger.annotations.ApiOperation;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.common.ws.jersey.RolesForbidden;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.proto.Segment;
import ru.yandex.crypta.lab.proto.SegmentAttributes;

@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class SegmentExportResource extends CommonLabResource {
    private final Provider<SecurityContext> securityContextProvider;

    private final Set<String> systemTags = Set.of(
            "adrush",
            "agusha",
            "aletcrm",
            "aquaminerale",
            "audience",
            "banana",
            "bbc",
            "biomax",
            "cheetos",
            "chudo",
            "chudodetki",
            "crypta_id",
            "direct",
            "display",
            "dit",
            "domikvderevne",
            "doritos",
            "driveme",
            "fruktsad",
            "hrusteam",
            "imunele",
            "j7",
            "kazakhstan",
            "lamber",
            "lays",
            "lipton",
            "lyubimiy",
            "mirinda",
            "naturerush",
            "pepsi",
            "plus",
            "potata",
            "trainable",
            "ya",
            "yandex.surveys",
            "не_экспортируемый"
    );

    private final List<String> systemTagPrefixes = List.of(
            "ca_type:",
            "ml_segments"
    );

    @Inject
    public SegmentExportResource(LabService lab, Provider<SecurityContext> securityContextProvider) {
        super(lab);
        this.securityContextProvider = securityContextProvider;
    }

    private boolean isSystemTag(String tag) {
        return systemTags.contains(tag) || systemTagPrefixes.stream().anyMatch(tag::startsWith);
    }

    @POST
    @ApiOperation(value = "Create new segment export")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment.Export createSegmentExport(
            @QueryParam("segmentId") @NotNull String segmentId,
            @QueryParam("type") @NotNull Segment.Export.Type type,
            @QueryParam("exportKeywordId") @NotNull Long exportKeywordId,
            @QueryParam("exportSegmentId") Long exportSegmentId,
            @QueryParam("exportCryptaId") Boolean exportCryptaId
    )
    {
        Segment.Export.Builder export = Segment.Export.newBuilder()
                .setType(type)
                .setKeywordId(exportKeywordId);
        Optional.ofNullable(exportSegmentId).ifPresent(export::setSegmentId);

        boolean uploadCryptaId = exportCryptaId != null ? exportCryptaId : false;
        return lab().segmentExports().createExport(segmentId, uploadCryptaId, export);
    }

    @POST
    @ApiOperation(value = "Create new user segment export")
    @Path("user")
    @RolesForbidden(Roles.Lab.SEGMENTS_NOT_ALLOWED_META_ROLE)
    public Segment.Export createUserSegmentExport(
            @QueryParam("segmentId") @NotNull String segmentId,
            @QueryParam("keywordId") @NotNull Integer keywordId
    )
    {
        Long kw = Long.valueOf(keywordId);
        if (!(kw.equals(557L) || kw.equals(549L))) {
            throw Exceptions.wrongRequestException("Unsupported keyword", "BAD_KEYWORD");
        }

        Segment.Export.Builder export = Segment.Export.newBuilder()
                .setType(Segment.Export.Type.HEURISTIC)
                .setKeywordId(kw);
        return lab().segmentExports().createExport(segmentId, false, export);
    }

    @PUT
    @ApiOperation(value = "Update segment export")
    @Path("{id}")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment.Export updateSegmentExport(
            @PathParam("id") String id,
            @QueryParam("type") Segment.Export.Type type,
            @QueryParam("exportKeywordId") Long exportKeywordId,
            @QueryParam("exportSegmentId") Long exportSegmentId
    )
    {
        Segment.Export.Builder export = Segment.Export.newBuilder();
        Optional.ofNullable(type).ifPresent(export::setType);
        Optional.ofNullable(exportKeywordId).ifPresent(export::setKeywordId);
        Optional.ofNullable(exportSegmentId).ifPresent(export::setSegmentId);
        return lab().segmentExports().updateExport(id, export);
    }

    private boolean allowedToEditExport(String id) {
        var segmentId = lab().segments().getByExportId(id).getId();
        return lab().segments().allowedToEditSegment(segmentId, securityContextProvider);
    }

    @DELETE
    @ApiOperation(value = "Delete segment export")
    @Path("{id}")
    public Segment.Export deleteSegmentExport(
            @PathParam("id") String id
    )
    {
        if (!allowedToEditExport(id)) {
            throw Exceptions.forbidden(String.format(
                    "User %s is not allowed to edit export=%s. User must be an author or a responsible of the segment " +
                            "the export belongs to or to be in role \"lab_admin\" or \"lab_extended\".",
                    securityContextProvider.get().getUserPrincipal().getName(),
                    id
                    ),
                    "403");
        }
        return lab().segmentExports().deleteExport(id);
    }

    @GET
    @ApiOperation(value = "Get segment export")
    @Path("{id}")
    public Segment.Export getSegmentExport(
            @PathParam("id") String id
    )
    {
        return lab().segmentExports().getExport(id);
    }

    @GET
    @ApiOperation(value = "Get segment by export id")
    @Path("{id}/segment")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public SegmentAttributes getSegmentByExportId(
            @PathParam("id") @NotNull String id
    ) {
        return lab().segments().getByExportId(id);
    }

    @GET
    @ApiOperation(value = "Get user_set_id by export_id")
    @Path("get_user_set_id_by_export_id")
    public String getUserSetIdByExportId(
            @QueryParam("export_id") @NotNull String exportId
    )
    {
        return lab().getUserSetIdByExportId(exportId);
    }


    @PUT
    @ApiOperation(value = "Bind segment export")
    @Path("{id}/bind")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment.Export bindSegmentExport(
            @PathParam("id") String id,
            @QueryParam("segmentId") @NotNull String segmentId
    )
    {
        return lab().segmentExports().bindExport(id, segmentId);
    }

    @PUT
    @ApiOperation(value = "Update bigb coverage")
    @Path("{id}/coverage/bigb")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment.Export updateExportBigbCoverage(
            @PathParam("id") String id,
            @QueryParam("value") @NotNull Long value,
            @QueryParam("timestamp") @NotNull Long timestamp
    )
    {
        return lab().segmentExports().updateExportBigbCoverage(id, value, timestamp);
    }

    @PUT
    @ApiOperation(value = "Update profiles coverage")
    @Path("{id}/coverage/profiles")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment.Export updateExportProfilesCoverage(
            @PathParam("id") String id,
            @QueryParam("value") @NotNull Long value,
            @QueryParam("timestamp") @NotNull Long timestamp
    )
    {
        return lab().segmentExports().updateExportProfilesCoverage(id, value, timestamp);
    }

    @PUT
    @ApiOperation(value = "Update export state")
    @Path("{id}/state")
    public Segment.Export updateExportState(
            @PathParam("id") String id,
            @QueryParam("state") @NotNull Segment.Export.State state
    )
    {
        if (!allowedToEditExport(id)) {
            throw Exceptions.forbidden(String.format(
                    "User %s is not allowed to edit export=%s.",
                    securityContextProvider.get().getUserPrincipal().getName(),
                    id
            ),
            "403");
        }

        Segment.Export.State exportState = getSegmentExport(id).getState();

        if (!(exportState == Segment.Export.State.DISABLE_REQUESTED && state == Segment.Export.State.ACTIVE) && !securityContextProvider.get().isUserInRole(Roles.Lab.ADMIN)) {
            throw Exceptions.forbidden(String.format(
                    "User %s is not allowed to edit export=%s state from=%s to=%s",
                    securityContextProvider.get().getUserPrincipal().getName(),
                    id,
                    exportState.getValueDescriptor().getName(),
                    state
            ),
            "403");
        }

        return lab().segmentExports().updateExportState(id, state);
    }

    @PUT
    @ApiOperation(value = "Put export expression")
    @Consumes(MediaType.APPLICATION_FORM_URLENCODED)
    @Path("{id}/expression")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN, Roles.Lab.EXPRESSIONS_EDITOR})
    public Segment.Export putExportExpression(
            @PathParam("id") String id,
            @FormParam("expressions") @NotNull List<String> expressions
    )
    {
        return lab().segmentExports().putExportExpression(id, expressions);
    }

    @DELETE
    @ApiOperation(value = "Delete export expression")
    @Path("{id}/expression")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN, Roles.Lab.EXPRESSIONS_EDITOR})
    public Segment.Export deleteExportExpression(
            @PathParam("id") String id
    )
    {
        return lab().segmentExports().deleteExportExpression(id);
    }

    @PUT
    @ApiOperation(value = "Put lal")
    @Consumes(MediaType.APPLICATION_FORM_URLENCODED)
    @Path("{id}/lal")
    @RolesForbidden(Roles.Lab.SEGMENTS_NOT_ALLOWED_META_ROLE)
    public Segment.Export putLal(
            @PathParam("id") String id,
            @FormParam("lal") @NotNull String lal
    )
    {
        return lab().segmentExports().putLal(id, lal);
    }

    @DELETE
    @ApiOperation(value = "Delete lal")
    @Path("{id}/lal")
    @RolesForbidden(Roles.Lab.SEGMENTS_NOT_ALLOWED_META_ROLE)
    public Segment.Export deleteLal(
            @PathParam("id") String id
    )
    {
        return lab().segmentExports().deleteLal(id);
    }

    @PUT
    @ApiOperation(value = "Put export rule id")
    @Path("{id}/rule_id")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment.Export putExportRuleId(
            @PathParam("id") String id,
            @QueryParam("ruleId") @NotNull String ruleId
    )
    {
        return lab().segmentExports().putExportRuleId(id, ruleId);
    }

    @DELETE
    @ApiOperation(value = "Delete export rule id")
    @Path("{id}/rule_id")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment.Export deleteExportRuleId(
            @PathParam("id") String id
    )
    {
        return lab().segmentExports().deleteExportRuleId(id);
    }

    @GET
    @ApiOperation(value = "Get all exports with rule id")
    @Path("rule_id")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public List<Segment.Export> getExportsWithRuleId()
    {
        return lab().segmentExports().getExportsWithRuleId();
    }

    @GET
    @ApiOperation(value = "Get available segment id by keyword id")
    @Path("segment_id/available/{keyword_id}")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Long getAvailableSegmentId(
            @PathParam("keyword_id") @NotNull Long keywordId
    )
    {
        return lab().segmentExports().getAvailableSegmentId(keywordId);
    }

    @GET
    @ApiOperation(value = "Get list of not exported to BigB exports")
    @Path("exports_not_for_bigb")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public List<Segment.Export> getNotExportedToBigB() {
        return lab().segmentExports().getExportsNotExportedToBigB();
    }

    @PUT
    @ApiOperation(value = "Disable export to BigB")
    @Path("{id}/disable_export_to_bigb")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment.Export disableExportToBigB(
            @PathParam("id") @NotNull String exportId
    ) {
        return lab().segmentExports().disableExportToBigB(exportId);
    }

    @PUT
    @ApiOperation(value = "Enable export to BigB")
    @Path("{id}/enable_export_to_bigb")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment.Export enableExportToBigB(
            @PathParam("id") @NotNull String exportId
    ) {
        return lab().segmentExports().enableExportToBigB(exportId);
    }

    @POST
    @Path("tags/{id}")
    @ApiOperation(value = "Add tag of segment export")
    public Segment.Tags addTag(
            @PathParam("id") @NotNull String segmentExportId,
            @QueryParam("tag") @NotNull String tag
    ) {
        if (isSystemTag(tag) && !securityContextProvider.get().isUserInRole(Roles.Lab.ADMIN)) {
            throw Exceptions.forbidden(
                    String.format(
                            "User %s does not have roles to add system tag \"%s\"",
                            securityContextProvider.get().getUserPrincipal().getName(),
                            tag
                    ),
                    "WRONG_ROLE"
            );
        }

        return lab().segmentExports().addTag(segmentExportId, tag);
    }

    @GET
    @Path("tags/{id}")
    @ApiOperation(value = "Tag lists of segment export")
    public Segment.Tags getTags(
            @PathParam("id") @NotNull String segmentExportId
    ) {
        return lab().segmentExports().getTags(segmentExportId);
    }

    @DELETE
    @Path("tags/{id}")
    @ApiOperation(value = "Delete tag of segment export")
    public Segment.Tags deleteTag(
            @PathParam("id") @NotNull String segmentExportId,
            @QueryParam("tag") @NotNull String tag
    ) {
        if (isSystemTag(tag) && !securityContextProvider.get().isUserInRole(Roles.Lab.ADMIN)) {
            throw Exceptions.forbidden(
                    String.format(
                            "User %s does not have roles to delete system tag \"%s\"",
                            securityContextProvider.get().getUserPrincipal().getName(),
                            tag
                    ),
                    "WRONG_ROLE"
            );
        }

        return lab().segmentExports().removeTag(segmentExportId, tag);
    }

    @PUT
    @ApiOperation(value = "Update export next activity check timestamp")
    @Path("{id}/update_next_activity_check_ts")
    public Segment.Export updateExportNextActivityCheckTs(
            @PathParam("id") @NotNull String exportId,
            @QueryParam("timestamp") @NotNull Long timestamp
    ) {
        if (!allowedToEditExport(exportId)) {
            throw Exceptions.forbidden(String.format(
                    "User %s is not allowed to edit export=%s.",
                    securityContextProvider.get().getUserPrincipal().getName(),
                    exportId
            ),
            "403");
        }
        return lab().segmentExports().updateExportNextActivityCheckTs(exportId, timestamp);
    }

    @Path("audiences")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Class<SegmentExportAudienceResource> audienceGrants() {
        return SegmentExportAudienceResource.class;
    }

}
