package ru.yandex.crypta.api.rest.resource.lab;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.inject.Provider;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.FormParam;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.SecurityContext;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.util.JsonFormat;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.common.ws.jersey.RolesForbidden;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lab.I18Utils;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.proto.Model;
import ru.yandex.crypta.lab.proto.Segment;
import ru.yandex.crypta.lab.proto.SegmentAttributes;
import ru.yandex.crypta.lab.proto.SegmentConditions;
import ru.yandex.crypta.lab.proto.SegmentGroup;
import ru.yandex.crypta.lab.proto.SegmentIdPair;
import ru.yandex.crypta.lab.proto.TSimpleSampleStatsWithInfo;
import ru.yandex.crypta.lab.proto.Translations;
import ru.yandex.crypta.lab.utils.SegmentName;
import ru.yandex.crypta.lab.utils.SegmentNode;
import ru.yandex.inside.yt.kosher.cypress.YPath;

@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class SegmentResource extends CommonLabResource {
    private final Provider<SecurityContext> securityContextProvider;

    @Inject
    public SegmentResource(LabService lab, Provider<SecurityContext> securityContextProvider) {
        super(lab);
        this.securityContextProvider = securityContextProvider;
    }

    @GET
    @ApiOperation(value = "Get all segments")
    public List<Segment> getAllSegments() {
        return lab().segments().getAll();
    }

    @GET
    @Path("attributes")
    @ApiOperation(value = "Get segments attributes")
    public List<SegmentAttributes> getSegmentsAttributes() {
        return lab().segments().getSegmentsAttributes();
    }

    @GET
    @Path("match_search_items")
    @ApiOperation(value = "Matches segment fields with search items")
    public List<SegmentAttributes> matchSegmentsSearchItems(@NotNull @QueryParam("searchItems") List<String> searchItems) {
        return lab().segments().matchSearchItems(searchItems);
    }

    @GET
    @Path("match_conditions")
    @ApiOperation(value = "Match segments with conditions and retrieve ones")
    public List<SegmentAttributes> matchSegmentsWithConditions(@NotNull @QueryParam("conditions") String conditions) {
        SegmentConditions.Builder conditionsObject = SegmentConditions.newBuilder();

        try {
            JsonFormat.parser().ignoringUnknownFields().merge(conditions, conditionsObject);
        } catch (InvalidProtocolBufferException e) {
            throw Exceptions.wrongRequestException("Unsupported condition values", "BAD_PARAMETERS");
        }

        return lab().segments().matchSegmentConditions(conditionsObject.build());
    }

    @GET
    @Path("{id}")
    @ApiOperation(value = "Get segment")
    public Segment getSegment(@NotNull @PathParam("id") String id) {
        return lab().segments().get(id);
    }

    @GET
    @Path("{id}/notLocalized")
    @ApiOperation(value = "Get not localized segment")
    public Segment getNotLocalizedSegment(@NotNull @PathParam("id") String id) {
        return lab().segments().getNotLocalized(id);
    }

    @GET
    @Path("{id}/activeModels")
    @ApiOperation(value = "Get active models")
    public List<Model> getActiveModelsOfSegment(@PathParam("id") String id) {
        return lab().models().getActive(id);
    }

    @GET
    @Path("getDirectTaxonomy")
    @ApiOperation(value = "Get user profile taxonomy for Yandex Direct frontend")
    @Deprecated
    public JsonNode getDirectTaxonomy() {
        return lab().getDirectTaxonomy();
    }


    @DELETE
    @Path("{id}")
    @ApiOperation(value = "Delete segment")
    public Segment deleteSegment(@PathParam("id") String id) {
        if (!lab().segments().allowedToEditSegment(id, securityContextProvider)) {
            throw Exceptions.forbidden(
                    String.format(
                            "User %s is not allowed to delete segment=%s. User must be an author or a responsible of the segment " +
                                    "or to be in role \"lab_admin\" or \"lab_extended\".",
                            securityContextProvider.get().getUserPrincipal().getName(),
                            id
                    ),
                    "403");
        }
        return lab().segments().deleteSegment(id);
    }

    @POST
    @ApiOperation(value = "Create new segment")
    @Consumes(MediaType.APPLICATION_FORM_URLENCODED)
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment createSegment(
            @FormParam("preferredId") @ApiParam("Preferred id suffix, \"segment-[preferedId]\"") String preferredId,
            @FormParam("nameEn") @ApiParam("Name") @DefaultValue("") String nameEn,
            @FormParam("nameRu") @ApiParam("Название") @NotBlank String nameRu,
            @FormParam("descriptionEn") @ApiParam("Description") @DefaultValue("") String descriptionEn,
            @FormParam("descriptionRu") @ApiParam("Описание") @NotBlank String descriptionRu,
            @FormParam("tickets") @ApiParam("Task id, e.g. \"CRYPTA-123\"") @NotEmpty List<String> tickets,
            @FormParam("scope") @NotNull Segment.Scope scope,
            @FormParam("type") @NotNull Segment.Type type,
            @FormParam("responsibles") @ApiParam("Email, e.g. \"login@yandex-team.ru\"") @NotNull List<String> responsibles,
            @FormParam("stakeholders") @ApiParam("Email, e.g. \"login@yandex-team.ru\"") @NotNull List<String> stakeholders,
            @FormParam("parentId") @NotBlank String parentId
    ) {
        Segment.Builder prototype = Segment
                .newBuilder()
                .setName(I18Utils.pack(I18Utils.string(nameEn, nameRu)))
                .setDescription(I18Utils.pack(I18Utils.string(descriptionEn, descriptionRu)))
                .addAllTickets(tickets)
                .setScope(scope)
                .setType(type)
                .addAllResponsibles(responsibles)
                .addAllStakeholders(stakeholders)
                .setParentId(parentId);

        return lab().segments().createSegment(preferredId, prototype);
    }

    @POST
    @Path("user")
    @ApiOperation(value = "Create new user segment")
    @Consumes(MediaType.APPLICATION_FORM_URLENCODED)
    @RolesForbidden(Roles.Lab.SEGMENTS_NOT_ALLOWED_META_ROLE)
    public Segment createUserSegment(
            @FormParam("nameEn") @DefaultValue("") String nameEn,
            @FormParam("nameRu") @NotBlank String nameRu,
            @FormParam("descriptionEn") @DefaultValue("") String descriptionEn,
            @FormParam("descriptionRu") @NotBlank String descriptionRu,
            @FormParam("tickets") @NotEmpty List<String> tickets
    ) {
        Segment.Builder segment = Segment
                .newBuilder()
                .setName(I18Utils.pack(I18Utils.string(nameEn, nameRu)))
                .setDescription(I18Utils.pack(I18Utils.string(descriptionEn, descriptionRu)))
                .addAllTickets(tickets);
        return lab().segments().createUserSegment(segment);
    }

    @GET
    @ApiOperation(value = "Get segment groups")
    @Path("groups")
    public List<SegmentGroup> getAllSegmentGroups() {
        return lab().segments().getAllGroups();
    }

    @GET
    @ApiOperation(value = "Get segment group by id")
    @Path("groups/{id}")
    public SegmentGroup getSegmentGroup(
            @PathParam("id") @NotNull String id
    ) {
        return lab().segments().getGroup(id);
    }

    @GET
    @Path("groups/{id}/notLocalized")
    @ApiOperation(value = "Get not localized segment group by id")
    public SegmentGroup getNotLocalizedGroup(@NotNull @PathParam("id") String id) {
        return lab().segments().getGroup(id, false);
    }

    @DELETE
    @ApiOperation(value = "Delete segment group by id")
    @Path("groups/{id}")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public SegmentGroup deleteSegmentGroup(
            @PathParam("id") @NotNull String id
    ) {
        return lab().segments().deleteGroup(id);
    }

    @POST
    @ApiOperation(value = "Create new Segment Group")
    @Path("groups")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public SegmentGroup createSegmentGroup(
            @QueryParam("nameEn") @DefaultValue("") String nameEn,
            @QueryParam("nameRu") @NotBlank String nameRu,
            @QueryParam("descriptionEn") @DefaultValue("") String descriptionEn,
            @QueryParam("descriptionRu") @NotBlank String descriptionRu,
            @QueryParam("parentId") String parentId
    ) {
        SegmentGroup.Builder prototype = SegmentGroup
                .newBuilder()
                .setName(I18Utils.pack(I18Utils.string(nameEn, nameRu)))
                .setDescription(I18Utils.pack(I18Utils.string(descriptionEn, descriptionRu)))
                .setParentId(Optional.ofNullable(parentId).orElse("root"));

        return lab().segments().createGroup(prototype);
    }

    @GET
    @ApiOperation(value = "Get segment groups as tree")
    @Path("groups_tree")
    public SegmentNode getSegmentGroupsTree() {
        return lab().segments().getGroupsTree();
    }

    @POST
    @Path("set_segment_group")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment setSegmentGroup(
            @QueryParam("segmentId") @NotNull String segmentId,
            @QueryParam("newGroupId") @NotNull String newGroupId
    ) {
        return lab().segments().setSegmentParent(segmentId, newGroupId);
    }

    @POST
    @Path("set_group_parent_id")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public SegmentGroup setGroupParentId(
            @QueryParam("groupId") String groupId,
            @QueryParam("newParentGroupId") String newParentGroupId
    ) {
        return lab().segments().setGroupParent(groupId, newParentGroupId);
    }

    @Deprecated
    @PUT
    @ApiOperation(value = "Update segment name")
    @Path("{id}/name")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment updateName(
            @PathParam("id") @NotNull String id,
            @QueryParam("value") @NotBlank String value
    ) {
        return lab().segments().updateName(id, value);
    }

    @POST
    @ApiOperation(value = "Update segments tanker keys in database")
    @Path("tanker/update_keys")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public void updateKeysFromTanker() {
        lab().segments().updateKeysFromTanker();
    }

    @GET
    @ApiOperation(value = "Get Tanker keys inconsistency")
    @Path("tanker/inconsistency")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public JsonNode getTankerKeysInconsistency() {
        return lab().segments().getTankerKeysInconsistency();
    }

    @DELETE
    @ApiOperation(value = "Remove Tanker key and translations from Tanker and Database")
    @Path("tanker/remove_key")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Translations removeKeyAndTranslations(String key) {
        return lab().segments().removeKeyAndTranslations(key);
    }

    @GET
    @ApiOperation(value = "Get translations from Tanker by key")
    @Path("tanker/{key}")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Translations getTranslationsByKey(@PathParam("key") String tankerKey) {
        return lab().segments().getTranslationsFromTankerByKey(tankerKey);
    }

    @GET
    @ApiOperation(value = "Get all tanker translations")
    @Path("tanker")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public JsonNode getAllTranslations() {
        return lab().segments().getTranslations();
    }

    @PUT
    @ApiOperation(value = "Creates keys for name and descriptions in Tanker")
    @Path("tanker/create_segment_keys")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Map<String, Translations> createSegmentKeysInTanker(
            @QueryParam("id") @NotNull String id,
            @QueryParam("name") @NotNull String name,
            @QueryParam("description") @NotNull String description,
            @QueryParam("skipTranslation") @NotNull @DefaultValue("false") Boolean skipTranslation
    ) {
        return lab().segments().createSegmentKeysInTanker(id, name, description, skipTranslation);
    }

    @DELETE
    @ApiOperation(value = "Removes name and description from database and tanker")
    @Path("tanker/delete_segment_keys")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Map<String, Translations> deleteSegmentKeysInTanker(
            @QueryParam("segmentId") @NotNull String segmentId
    ) {
        return lab().segments().deleteSegmentKeysFromTanker(segmentId);
    }

    @Deprecated
    @PUT
    @ApiOperation(value = "Update segment description")
    @Path("{id}/description")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public Segment updateDescription(
            @PathParam("id") @NotNull String id,
            @QueryParam("value") @NotBlank String value
    ) {
        return lab().segments().updateDescription(id, value);
    }

    @PUT
    @ApiOperation(value = "Update segment name and description")
    @Path("{id}/name_and_description")
    public Segment updateNameAndDescription(
            @PathParam("id") @NotNull String id,
            @QueryParam("name") @NotBlank String name,
            @QueryParam("description") @NotBlank String description
    ) {
        return lab().segments().updateNameAndDescription(id, name, description);
    }

    @PUT
    @ApiOperation(value = "Update priority")
    @Path("{id}/priority")
    @RolesAllowed({Roles.Lab.EXTENDED, Roles.Lab.ADMIN})
    public void updatePriority(
            @PathParam("id") @NotNull String id,
            @QueryParam("priority") @NotNull Long priority
    ) {
        lab().segments().updatePriority(id, priority);
    }

    @PUT
    @ApiOperation(value = "Add a stakeholder")
    @Path("{id}/stakeholder")
    public void addStakeholder(
            @PathParam("id") @NotNull String segmentId,
            @QueryParam("stakeholder") @NotBlank String stakeholder
    ) {
        lab().segments().addStakeholder(segmentId, stakeholder);
    }

    @DELETE
    @ApiOperation(value = "Delete a stakeholder")
    @Path("{id}/stakeholder")
    public void deleteStakeholder(
            @PathParam("id") @NotNull String segmentId,
            @QueryParam("stakeholder") @NotNull String stakeholder
    ) {
        lab().segments().deleteStakeholder(segmentId, stakeholder);
    }

    @PUT
    @ApiOperation(value = "Add responsible")
    @Path("{id}/responsible")
    public void addResponsible(
            @PathParam("id") @NotNull String segmentId,
            @QueryParam("responsible") @NotBlank String responsible
    ) {
        lab().segments().addResponsible(segmentId, responsible);
    }

    @DELETE
    @ApiOperation(value = "Delete responsible")
    @Path("{id}/responsible")
    public void deleteResponsible(
            @PathParam("id") @NotNull String segmentId,
            @QueryParam("responsible") @NotNull String responsible
    ) {
        lab().segments().deleteResponsible(segmentId, responsible);
    }

    @GET
    @ApiOperation(value = "Retrieves stats")
    @Path("{id}/stats")
    public TSimpleSampleStatsWithInfo getSegmentStats(
            @ApiParam("Segment id") @PathParam("id") @NotNull String segmentId
    ) {
        return lab().segments().getStats(segmentId).orElseThrow(Exceptions::notFound);
    }

    @GET
    @ApiOperation(value = "Retrieves siberia stats")
    @Path("{id}/siberia_stats")
    public TSimpleSampleStatsWithInfo getSegmentSiberiaStats(
            @ApiParam("Export id") @PathParam("id") @NotNull String exportId,
            @ApiParam("Base export id") @QueryParam("base_export_id") String baseExportId,
            @ApiParam("Base sample id") @QueryParam("base_sample_id") String baseSampleId,
            @ApiParam("Base group id") @QueryParam("base_group_id") String baseGroupId
    ) {
        return lab().segments().getStats(
                exportId,
                Optional.ofNullable(baseExportId),
                Optional.ofNullable(baseSampleId),
                Optional.ofNullable(baseGroupId)
        ).orElseThrow(Exceptions::notFound);
    }

    @GET
    @ApiOperation(value = "Get transitive parents of segments")
    @Path("parents")
    public Map<String, List<String>> getParentsPerSegment() {
        Map<String, List<SegmentGroup>> parentsPerSegment = lab().segments().getParentsPerSegment();
        return parentsPerSegment.entrySet().stream().collect(Collectors.toMap(
                Map.Entry::getKey,
                each -> each.getValue().stream().map(SegmentGroup::getId).collect(Collectors.toList())
        ));
    }

    @GET
    @ApiOperation(value = "Get segment names by keyword id")
    @Path("names/{keyword_id}")
    @Deprecated
    public Map<Long, SegmentName> getSegmentNamesWithExportId(
            @PathParam("keyword_id") @NotNull Long keywordId
    ) {
        return lab().segments().getSegmentNamesWithExportId(keywordId);
    }

    @GET
    @ApiOperation(value = "Get user_set_id by segment_lab_id")
    @Path("{id}/get_user_set_id")
    public SegmentIdPair getSegmentIdPair(
            @PathParam("id") String segmentLabId
    ) {
        return lab().segments().getSegmentPairBySegmentLabId(segmentLabId);
    }

    @POST
    @ApiOperation(value = "Store segment_lab_id -> user_set_id mapping into PGaaS")
    @Path("update_segment_ids_into_base")
    public List<SegmentIdPair> updateSegmentIdsTable(
            @QueryParam("source") String sourceTablePath
    ) {
        return lab().segments().updateSegmentIdsTable(
                YPath.simple(sourceTablePath)
        );
    }

}
