package ru.yandex.crypta.api.rest.resource.lab;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;

import javax.inject.Inject;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import com.sun.istack.NotNull;
import io.swagger.annotations.ApiOperation;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.graph.Identifier;
import ru.yandex.crypta.lab.proto.Columns;
import ru.yandex.crypta.lab.proto.Rule;
import ru.yandex.crypta.lab.startrek.LabStartrekService;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.common.YtErrorMapping;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.startrek.client.model.Issue;

@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class UtilsResource {
    private final YtService yt;
    private final LabStartrekService startrek;

    private static final int SAMPLE_SIZE = 10;

    private static final Set<EIdType> VALID_ID_TYPES = Set.of(
            EIdType.EMAIL_MD5,
            EIdType.YANDEXUID,
            EIdType.EMAIL,
            EIdType.PHONE,
            EIdType.PHONE_MD5,
            EIdType.PUID,
            EIdType.LOGIN,
            EIdType.MM_DEVICE_ID,
            EIdType.UUID,
            EIdType.CRYPTA_ID,
            EIdType.DIRECT_CLIENT_ID,
            EIdType.IDFA_GAID
    );

    @Inject
    public UtilsResource(YtService yt, LabStartrekService startrek) {
        this.yt = yt;
        this.startrek = startrek;
    }

    @GET
    @Path("get_lab_segment_table_columns")
    @ApiOperation("Validate segment source table and get columns")
    public Set<String> getLabSegmentColumns(
            @QueryParam("path") String path
    ) {
        try {
            var tablePath = getYPath(path);
            return getColumns(tablePath);
        } catch (IllegalArgumentException e) {
            throw Exceptions.wrongRequestException("Corrupted YT path", "BAD_PATH");
        } catch (YtErrorMapping.ResolveError e) {
            throw Exceptions.wrongRequestException("No such table", "NOT_FOUND");
        } catch (YtErrorMapping.AuthorizationError e) {
            throw Exceptions.forbidden("No read permission", "CANT_READ");
        }
    }

    @GET
    @Path("get_lab_segment_table_columns_with_id_types")
    @ApiOperation("Validate segment source table and get columns with possible id types")
    public Columns getLabSegmentColumnsWithIdTypes(
            @QueryParam("path") String path,
            @QueryParam("minRows") Integer minRows
    ) {
        try {
            var tablePath = getYPath(path);
            var columns = getPossibleColumnIdTypes(tablePath);
            if (minRows != null && minRows > 0) {
                checkRowCount(tablePath, minRows);
            }
            return columns;
        } catch (IllegalArgumentException e) {
            throw Exceptions.wrongRequestException("Corrupted YT path", "BAD_PATH");
        } catch (YtErrorMapping.ResolveError e) {
            throw Exceptions.wrongRequestException("No such table", "NOT_FOUND");
        } catch (YtErrorMapping.AuthorizationError e) {
            throw Exceptions.forbidden("No read permission", "CANT_READ");
        }
    }

    private YPath getYPath(String path) {
        if (!path.startsWith("//")) {
            throw Exceptions.wrongRequestException("Corrupted YT path", "BAD_PATH");
        }

        if (path.endsWith("/")) {
            path = path.substring(0, path.length() - 1);
        }

        return YPath.simple(path);
    }

    private void checkRowCount(YPath tablePath, int minRows) {
        var rowsNumber = Long.parseLong(String.valueOf(yt.getHahn().cypress().get(tablePath.attribute("row_count"))));
        if (rowsNumber < minRows) {
            throw Exceptions.wrongRequestException("Not enough rows", "NOT_ENOUGH_ROWS");
        }
    }

    private Set<String> getColumns(YPath tablePath) {
        if (!yt.getHahn().cypress().get(tablePath.attribute("type")).stringValue().equals("table")) {
            throw Exceptions.wrongRequestException("Not a table", "NOT_TABLE");
        }

        var schema = yt.getHahn().cypress().get(tablePath.attribute("schema")).asList();

        if (schema.isEmpty()) {
            throw Exceptions.wrongRequestException("No schema", "NO_SCHEMA");
        }

        var columns = new HashSet<String>();
        schema.forEach(node -> Optional.ofNullable(node.asMap().get("name")).ifPresent(
                value -> columns.add(value.stringValue())
        ));

        if (columns.isEmpty()) {
            throw Exceptions.wrongRequestException("No columns", "NO_COLUMNS");
        }

        return columns;
    }

    private Columns getPossibleColumnIdTypes(YPath tablePath) {
        var columns = getColumns(tablePath);

        var sample = new ArrayList<YTreeMapNode>();
        yt.getHahn().tables().read(tablePath.withRange(0, SAMPLE_SIZE), YTableEntryTypes.YSON,
                (Consumer<YTreeMapNode>) sample::add);

        var columnIdInvalidTypes = new HashMap<String, Set<EIdType>>();
        for (var column : columns) {
            var types = new HashSet<>(VALID_ID_TYPES);
            columnIdInvalidTypes.put(column, types);
        }

        for (var row : sample) {
            for (var column : columnIdInvalidTypes.keySet()) {
                if (!row.containsKey(column)) {
                    continue;
                }
                var value = row.get(column).get();
                String stringValue = null;
                if (value.isIntegerNode()) {
                    stringValue = Long.toUnsignedString(value.longValue());
                } else if (value.isStringNode()) {
                    stringValue = value.stringValue();
                } else {
                    continue;
                }

                var invalidTypes = columnIdInvalidTypes.get(column);
                var validTypes = new HashSet<EIdType>();
                for (var type : invalidTypes) {
                    var id = new Identifier(type, stringValue);
                    if (id.isValid()) {
                        validTypes.add(type);
                    }
                }
                invalidTypes.removeAll(validTypes);
            }
        }

        var result = Columns.newBuilder();
        for (var column : columnIdInvalidTypes.keySet()) {
            var types = new HashSet<>(VALID_ID_TYPES);
            var invalidTypes = columnIdInvalidTypes.get(column);
            types.removeAll(invalidTypes);

            if (types.isEmpty()) {
                continue;
            }

            var columnBuilder = result.addColumnsBuilder().setName(column);
            for (var type : types) {
                columnBuilder.addIdTypes(ru.yandex.crypta.lab.Identifier.byIdType(type).getName());
            }
        }

        if (result.getColumnsCount() == 0) {
            throw Exceptions.wrongRequestException("No columns with valid id types", "NO_COLUMNS_WITH_VALID_ID_TYPES");
        }

        return result.build();
    }

    @POST
    @Path("request_rule_approve_via_ticket")
    @ApiOperation("Creates rule approval request in Startrek")
    public Rule requestRuleApproveViaTicket(
            @QueryParam("ruleId") @NotNull String ruleId,
            @QueryParam("author") @NotNull String login,
            @QueryParam("linkedIssue") String linkedIssue,
            @QueryParam("url") @NotNull String url
    ) {
        return startrek.requestRuleApprovalViaTicket(ruleId, login, linkedIssue, url);
    }

    @POST
    @Path("comment_on_approve_rule_issue")
    @ApiOperation("Comments new condition to approve and sets new status")
    public Issue commentOnApproveSegmentRuleIssue(
            @QueryParam("issueId") @NotNull String issueId,
            @QueryParam("comment") @NotNull String comment
    ) {
        return startrek.commentOnApproveSegmentRuleIssue(issueId, comment);
    }
}
