package ru.yandex.crypta.api.rest.resource.lab.grab;

import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.Optional;
import java.util.UUID;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.SecurityContext;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.BooleanNode;
import io.swagger.annotations.ApiOperation;

import ru.yandex.crypta.api.rest.resource.portal.grab.Base58;
import ru.yandex.crypta.api.rest.resource.portal.grab.GrabId;
import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Roles;

@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class AudienceGrabResource {
    private final AudienceGrabTable table;

    @Context
    SecurityContext securityContext;

    @Inject
    public AudienceGrabResource(PostgresClient postgresClient) {
        this.table = new AudienceGrabTable(postgresClient.getJooqConfiguration());
    }

    private static void validateGrabId(GrabId grabId) {
        Optional<String> grabIdDecoded = Base58.decodeCheck(grabId.getValue());
        if (!grabIdDecoded.isPresent()) {
            throw Exceptions.illegal("Invalid grab id");
        }
    }

    @POST
    @ApiOperation(value = "Create and store user session")
    public AudienceGrab createGrab(@QueryParam("ext_login") @NotNull String extLogin) {
        String login = securityContext.getUserPrincipal().getName();
        UUID uuid = UUID.randomUUID();
        String raw = String.join("|", login, extLogin, uuid.toString());
        // we obfuscate it a bit
        GrabId grabId = new GrabId(Base58.encodeCheck(raw.getBytes(StandardCharsets.US_ASCII)));
        validateGrabId(grabId);

        long timestamp = Instant.now().getEpochSecond();
        AudienceGrab grab = new AudienceGrab(grabId, timestamp, false, extLogin);

        table.upsertQuery(grab).execute();
        return grab;
    }

    @GET
    @ApiOperation(value = "Check if authorization is still on")
    public BooleanNode checkAudienceAuth(
            @QueryParam("grab_id") @NotNull String grabId,
            @QueryParam("ext_login") @NotNull String extLogin
    )
    {
        validateGrabId(new GrabId(grabId));
        Optional<AudienceGrab> grab = table.selectByIdQuery(grabId).fetchOptionalInto(AudienceGrab.class);
        if (grab.isPresent()) {
            return new ObjectMapper().convertValue(
                    grab.get().getLogin().equals(extLogin) && !grab.get().hasExpired(60 * 5) && grab.get().getAuth(),
                    BooleanNode.class
            );
        } else {
            throw Exceptions.notFound();
        }
    }

    @POST
    @Path("confirm_auth")
    @ApiOperation(value = "Confirm user auth")
    @RolesAllowed({Roles.Portal.GRAB})
    public void authorizeAudienceUser(
            @QueryParam("grab_id") @NotNull String grabId
    )
    {
        Optional<AudienceGrab> grabO = table.selectByIdQuery(grabId).fetchOptionalInto(AudienceGrab.class);
        if (grabO.isPresent()) {
            try {
                table.upsertQuery(grabO.get().withAuth(true)).execute();
            } catch (Exception e) {
                throw Exceptions.unavailable();
            }
        } else {
            throw Exceptions.notFound();
        }
    }

    @GET
    @Path("{grab_id}")
    @ApiOperation(value = "Get grab by id")
    public AudienceGrab getGrab(@PathParam("grab_id") @NotNull String grabId) {
        validateGrabId(new GrabId(grabId));
        Optional<AudienceGrab> grab = table.selectByIdQuery(grabId).fetchOptionalInto(AudienceGrab.class);
        if (grab.isPresent()) {
            return grab.get();
        } else {
            throw Exceptions.notFound();
        }
    }
}
