package ru.yandex.crypta.api.rest.resource.ltp;


import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.inject.Provider;
import javax.validation.constraints.NotNull;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.SecurityContext;

import com.google.common.base.Splitter;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;

import ru.yandex.crypta.api.proto.TLtpProfileRequestHistory;
import ru.yandex.crypta.api.proto.TLtpProfileResultResponse;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.ltp.viewer.services.api.proto.TGetHistoryRequest;
import ru.yandex.crypta.ltp.viewer.services.api.proto.TGetProgressResponse;
import ru.yandex.crypta.service.ltp.client.LtpViewerClient;

@Path("ltp")
@Api(tags = {"ltp"})
@Produces(JsonUtf8.MEDIA_TYPE)
@RolesAllowed({Roles.Portal.LONG_TERM_PROFILE})
public class LtpProfileResource {

    private final Provider<SecurityContext> securityContextProvider;
    private final LtpViewerClient ltpViewerClient;

    @Inject
    public LtpProfileResource(Provider<SecurityContext> securityContextProvider, LtpViewerClient ltpViewerClient) {
        this.securityContextProvider = securityContextProvider;
        this.ltpViewerClient = ltpViewerClient;
    }

    @GET
    @Path("{ResultID}")
    @ApiOperation(value = "Gets all user related events from Long-term-profile storage")
    public TLtpProfileResultResponse getLtpProfile(
            @PathParam("ResultID") @NotNull String resultID,
            @QueryParam("FromEventTs") int fromEventTs,
            @QueryParam("ToEventTs") int toEventTs,
            @QueryParam("IdFilter") @DefaultValue("") String idFilter,
            @QueryParam("IdTypeFilter") @DefaultValue("") String idTypeFilter,
            @QueryParam("SourceTypeFilter") @DefaultValue("") String sourceTypeFilter,
            @QueryParam("DescriptionFilter") @DefaultValue("") String descriptionFilter,
            @QueryParam("AdditionalDescriptionFilter") @DefaultValue("") String additionalDescriptionFilter,
            @QueryParam("Offset") @DefaultValue("0") int offset,
            @QueryParam("Limit") @DefaultValue("100") int limit,
            @QueryParam("Version") @DefaultValue("1") int version,
            @QueryParam("FromDate") @DefaultValue("") String fromDate,
            @QueryParam("ToDate") @DefaultValue("") String toDate
    ) {
        var resultIDParts = Splitter.on(":").split(resultID).iterator();
        var idType = resultIDParts.next();
        var id = resultIDParts.next();
        var owner = securityContextProvider.get().getUserPrincipal().getName();

        var request = TGetHistoryRequest.newBuilder();
        request.getIdBuilder()
                .setType(idType)
                .setValue(id);
        request.getFilterBuilder()
                .setMinTimestamp(fromEventTs)
                .setMaxTimestamp(toEventTs)
                .setIdFilter(idFilter)
                .setIdTypeFilter(idTypeFilter)
                .setLogFilter(sourceTypeFilter)
                .setDescriptionFilter(descriptionFilter)
                .setAdditionalDescriptionFilter(additionalDescriptionFilter);
        request.getPageBuilder().setLimit(limit).setOffset(offset);
        request.setOwner(owner)
               .setFromDate(fromDate)
               .setToDate(toDate);

        var response = TLtpProfileResultResponse.newBuilder()
                .setResultID(resultID)
                .setVersion(2)
                .setId(id)
                .setIdType(idType)
                .setReady(true)
                .setFromDate(fromDate)
                .setToDate(toDate);

        ltpViewerClient.getHistory(request.build()).ifPresent(
            history -> response.addAllItems(history.getItemsList()).setTotalRecs(history.getTotal())
        );

        return response.build();
    }

    @GET
    @Path("user_history")
    @ApiOperation(value = "Gets recent LTP profile requests for logged-in user")
    public TLtpProfileRequestHistory getUserHistory(
            @QueryParam("Version") @DefaultValue("1") int version
    ) {
        String login = securityContextProvider.get().getUserPrincipal().getName();
        var queries = ltpViewerClient.getUserQueries(login).orElseThrow();
        var response = TLtpProfileRequestHistory.newBuilder();

        for (var query : queries.getQueriesList()) {
            response.addItemsBuilder()
                    .setReady(query.getReady())
                    .setRequestTs(0)
                    .setResultID(String.format("%s:%s", query.getId().getType(), query.getId().getValue()))
                    .getOriginalRequestBuilder()
                    .setId(query.getId().getValue())
                    .setIdType(query.getId().getType())
                    .setOwner(login)
                    .setFromDate(query.getFromDate())
                    .setToDate(query.getToDate());
        }
        return response.setVersion(2).build();
    }

    @GET
    @Path("/progress")
    @ApiOperation(value = "Gets LTP completion progress")
    public TGetProgressResponse getLtpProgress(
            @QueryParam("Id") @NotNull String id,
            @QueryParam("IdType") @NotNull String idType,
            @QueryParam("FromDate") @DefaultValue("") String fromDate,
            @QueryParam("ToDate") @DefaultValue("") String toDate
    ) {
        return ltpViewerClient.getProgress(idType, id, fromDate, toDate).orElseThrow();
    }
}
