package ru.yandex.crypta.api.rest.resource.me;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.UUID;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.lib.schedulers.Schedulers;
import ru.yandex.crypta.lib.yt.PathUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.proto.Me;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.common.YtException;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

@Path("me/geo_evaluation")
@Api(tags = {"me"})
@Produces(JsonUtf8.MEDIA_TYPE)
public class GeoEvaluationResource {

    private static final String TABLET_STATE_ATTR = "tablet_state";
    private final YtService yt;
    private YPath geoEvaluationOutput;
    private ScheduledExecutorService executorService;

    @Inject
    public GeoEvaluationResource(YtService yt, EEnvironment environment,
            Schedulers schedulers)
    {
        this.yt = yt;
        this.geoEvaluationOutput = YPath.simple("//home/crypta")
                .child(PathUtils.toPath(environment))
                .child("portal")
                .child("geo")
                .child("evaluations");
        this.executorService = schedulers.getExecutor();
    }

    @POST
    @ApiOperation(value = "Stores single geo evaluation to YT")
    @Path("store")
    @SuppressWarnings("FutureReturnValueIgnored")
    public Me.GeoEvaluation storeGeoEvaluationToYt(
            @QueryParam("yandexuid") @ApiParam(value = "Yandexuid") @NotNull String yandexuid,
            @QueryParam("latitude") @ApiParam(value = "Latitude") @NotNull Double lat,
            @QueryParam("longitude") @ApiParam(value = "Longitude") @NotNull Double lon,
            @QueryParam("state") @ApiParam(value = "User Evaluation") @NotNull Me.GeoEvaluation.State state,
            @QueryParam("location") @ApiParam(value = "Location type") @NotNull Me.GeoEvaluation.Location location,
            @QueryParam("timestamp") @ApiParam(value = "Timestamp of evaluation") @NotNull Long timestamp
    )
    {
        DateTimeFormatter dtf = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        LocalDate localDate = LocalDate.now();
        Me.GeoEvaluation evaluation;
        evaluation = Me.GeoEvaluation.newBuilder()
                .setId(UUID.randomUUID().toString())
                .setYandexuid(yandexuid)
                .setLat(lat)
                .setLon(lon)
                .setState(state)
                .setLocation(location)
                .setTimestamp(timestamp)
                .build();

        ListF<YTreeMapNode> evaluationEntry = Cf.arrayList();
        evaluationEntry.add(YTree.mapBuilder()
                .key("yandexuid").value(evaluation.getYandexuid())
                .key("latitude").value(evaluation.getLat())
                .key("longitude").value(evaluation.getLon())
                .key("state").value(evaluation.getState().toString())
                .key("location").value(evaluation.getLocation().toString())
                .key("timestamp").value(evaluation.getTimestamp())
                .key("date").value(dtf.format(localDate))
                .buildMap()
        );

        try {
            yt.getHahn().tables()
                    .insertRows(geoEvaluationOutput, false, false, YTableEntryTypes.YSON, evaluationEntry.iterator());
        } catch (YtException e) {
            if (!isMounted(geoEvaluationOutput)) {
                executorService.schedule(this::mountGeoEvaluationTable, 1, TimeUnit.MILLISECONDS);
            }
            throw Exceptions.unavailable();
        }

        return evaluation;
    }

    private boolean isMounted(YPath table) {
        YTreeNode node = yt.getHahn().cypress().get(table, Cf.set(TABLET_STATE_ATTR));
        String tabletState = node.getAttribute(TABLET_STATE_ATTR).get().stringValue();
        return tabletState.equals("mounted");
    }

    private boolean mountGeoEvaluationTable() {
        yt.getHahn().tables().mount(geoEvaluationOutput);
        return true;
    }

    @POST
    @ApiOperation(value = "Creates table")
    @Path("create")
    @RolesAllowed({Roles.Portal.PROFILE})
    public void createGeoEvaluationTable() {
        yt.getHahn().cypress().create(
                geoEvaluationOutput,
                CypressNodeType.TABLE, Cf.map("schema", YTree.builder().beginList()
                        .beginMap().key("name").value("yandexuid").key("type").value("string").endMap()
                        .beginMap().key("name").value("timestamp").key("type").value("int32").endMap()
                        .beginMap().key("name").value("latitude").key("type").value("double").endMap()
                        .beginMap().key("name").value("longitude").key("type").value("double").endMap()
                        .beginMap().key("name").value("location").key("type").value("string").endMap()
                        .beginMap().key("name").value("state").key("type").value("string").endMap()
                        .beginMap().key("name").value("date").key("type").value("string").endMap()
                        .endList()
                        .build(), "dynamic", YTree.booleanNode(true)
                )
        );

        mountGeoEvaluationTable();
    }
}
