package ru.yandex.crypta.api.rest.resource.other;

import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Response;

import NBSYeti.NExperimentParameters.Experiment;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.protobuf.util.JsonFormat;
import io.swagger.annotations.Api;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.graph.Yandexuid;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.service.experiments.ExperimentParameter;
import ru.yandex.crypta.service.experiments.ExperimentsService;

@Api(tags = "experiments")
@Path("experiments")
@Produces(JsonUtf8.MEDIA_TYPE)
public class ExperimentsResource {

    private final ExperimentsService experimentsService;

    @Inject
    public ExperimentsResource(ExperimentsService experimentsService) {
        this.experimentsService = experimentsService;
    }

    private static void validateYandexuid(Yandexuid id) {
        if (!id.isValid()) {
            throw Exceptions.wrongRequestException("Incorrect yandexuid", "BAD_YANDEXUID");
        }
    }

    @GET
    @RolesAllowed({Roles.Portal.EXPERIMENTS})
    public Experiment.TExperimentParameters getExperimentParameters(
            @QueryParam("id") @NotNull Yandexuid id,
            @QueryParam("ts") Long ts
    )
    {
        validateYandexuid(id);
        Long effectiveTs = Objects.requireNonNullElse(ts, Instant.now().getEpochSecond());

        return experimentsService.getExperimentParameters(Long.parseUnsignedLong(id.getValue()), effectiveTs);
    }

    @GET
    @Path("active")
    @RolesAllowed({Roles.Portal.EXPERIMENTS})
    public Response getActiveExperiments(
            @QueryParam("id") @NotNull Yandexuid id,
            @QueryParam("ts") Long ts
    )
    {
        validateYandexuid(id);
        Long effectiveTs = Objects.requireNonNullElse(ts, Instant.now().getEpochSecond());
        List<NExperiments.Experiment.TExperiment> experiments =
                experimentsService.getActiveExperiments(Long.parseUnsignedLong(id.getValue()), effectiveTs);

        List<JsonNode> experimentsList = new ArrayList<>();
        ObjectMapper objectMapper = new ObjectMapper();
        experiments.forEach(experimentMessage -> {
            if (experimentMessage.getExperimentID() != 3) {
                try {
                    JsonNode parameters =
                            objectMapper.readTree(JsonFormat.printer().print(experimentMessage.getParameters()));
                    JsonNode experiment = objectMapper
                            .readTree(JsonFormat.printer().includingDefaultValueFields().print(experimentMessage));
                    experimentsList.add(((ObjectNode) experiment).set("Parameters", parameters));
                } catch (IOException e) {
                    throw Exceptions.unchecked(e);
                }
            }
        });

        return Response.ok(experimentsList).build();
    }

    @GET
    @Path("flatten")
    @RolesAllowed({Roles.Portal.EXPERIMENTS})
    public List<ExperimentParameter> getExperimentsParametersFlatten(
            @QueryParam("id") @NotNull Yandexuid id,
            @QueryParam("ts") Long ts
    )
    {
        validateYandexuid(id);
        Long effectiveTs = Objects.requireNonNullElse(ts, Instant.now().getEpochSecond());

        return experimentsService.getExperimentParametersFlatten(Long.parseUnsignedLong(id.getValue()), effectiveTs);
    }

}
