package ru.yandex.crypta.api.rest.resource.other;

import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.TriggerKey;
import org.quartz.impl.matchers.GroupMatcher;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.auth.AuthNotRequired;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lib.schedulers.JobView;
import ru.yandex.crypta.lib.schedulers.Schedulers;
import ru.yandex.crypta.lib.schedulers.TriggerView;

@Path("scheduler")
@Produces(JsonUtf8.MEDIA_TYPE)
@Api(tags = "scheduler")
public class SchedulerResource {

    private final Scheduler scheduler;

    @Inject
    public SchedulerResource(Schedulers schedulers) {
        this.scheduler = schedulers.getQuartz().get();
    }

    @Path("job")
    @GET
    @RolesAllowed({Roles.ADMIN})
    @ApiOperation("Retrieve keys of all jobs")
    public List<JobView> getAllJobs() {
        return doUnchecked(scheduler -> scheduler
                .getJobKeys(GroupMatcher.anyJobGroup())
                .stream()
                .map(jobKey -> doUnchecked(s -> s.getJobDetail(jobKey)))
                .filter(Objects::nonNull)
                .map(JobView::new)
                .collect(Collectors.toList())
        );
    }

    @Path("trigger")
    @GET
    @RolesAllowed({Roles.ADMIN})
    @ApiOperation("Retrieve keys of all job triggers")
    public List<TriggerView> getAllTriggers() {
        return doUnchecked(scheduler -> scheduler
                .getTriggerKeys(GroupMatcher.anyTriggerGroup())
                .stream()
                .map(triggerKey -> doUnchecked(s -> s.getTrigger(triggerKey)))
                .filter(Objects::nonNull)
                .map(TriggerView::new)
                .collect(Collectors.toList())
        );
    }

    @Path("check/no_failed_triggers")
    @GET
    @AuthNotRequired
    @ApiOperation("Check if scheduler has no failed triggers")
    @Produces(MediaType.TEXT_PLAIN)
    public Response checkNoFailedTriggers() {
        Set<TriggerKey> triggers = doUnchecked(scheduler -> scheduler.getTriggerKeys(GroupMatcher.anyTriggerGroup()));
        for (TriggerKey triggerKey : triggers) {
            Trigger.TriggerState triggerState = getTriggerState(triggerKey);
            if (triggerState.equals(Trigger.TriggerState.ERROR)) {
                return Response.serverError().entity(triggerKey.toString()).build();
            }
        }
        return Response.ok("OK").build();
    }

    private Trigger.TriggerState getTriggerState(TriggerKey triggerKey) {
        return doUnchecked(scheduler -> scheduler.getTriggerState(triggerKey));
    }

    private <T> T doUnchecked(CheckedSchedulerOperation<T> function) {
        try {
            return function.apply(scheduler);
        } catch (SchedulerException e) {
            throw Exceptions.unchecked(e);
        }
    }

    private interface CheckedSchedulerOperation<T> {
        T apply(Scheduler scheduler) throws SchedulerException;
    }

}
