package ru.yandex.crypta.api.rest.resource.portal;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.function.Consumer;

import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.graph.Yandexuid;
import ru.yandex.crypta.lib.yt.YtReadingUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;

@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class GeoResource {

    private final Yt yt;

    @Inject
    public GeoResource(YtService ytService) {
        this.yt = ytService.getHahn();
    }

    private YPath homeworkRoot() {
        return YPath.cypressRoot()
                .child("home")
                .child("user_identification")
                .child("homework")
                .child("prod")
                .child("state");
    }

    private YPath latestHomeworkRoot() {
        return yt.cypress()
                .list(homeworkRoot())
                .stream()
                .map(YTreeStringNode::getValue)
                .map(value -> homeworkRoot().child(value))
                .max(Comparator.comparing(YPath::toString))
                .orElseThrow();
    }

    private YPath latestHomeworkByYuidTable() {
        return latestHomeworkRoot().child("homework_yuid");
    }

    private Location readLocation(String yandexuid, String key) {
        YPath path = latestHomeworkByYuidTable().withExact(YtReadingUtils.exact(YTree.stringNode(yandexuid)));
        List<YTreeMapNode> records = new ArrayList<>(1);
        yt.tables().read(path, YTableEntryTypes.YSON, (Consumer<YTreeMapNode>) records::add);
        if (records.isEmpty()) {
            throw Exceptions.notFound();
        }
        YTreeMapNode value = records.get(0).get(key).get().mapNode();
        double lat = value.getDouble("latitude");
        double lon = value.getDouble("longitude");
        double weight = value.getDouble("weight");
        return new Location(lat, lon, weight);
    }

    @GET
    @Path("home")
    @ApiOperation(value = "Get location of home")
    public Location getHome(@QueryParam("yandexuid") @ApiParam(value = "Yandexuid") @NotNull Yandexuid yandexuid) {
        return readLocation(yandexuid.toString(), "predicted_home");
    }

    @GET
    @Path("work")
    @ApiOperation(value = "Get location of work")
    public Location getWork(@QueryParam("yandexuid") @ApiParam(value = "Yandexuid") @NotNull Yandexuid yandexuid) {
        return readLocation(yandexuid.toString(), "predicted_work");
    }

    public static class Location {

        private final double lat;
        private final double lon;
        private final double weight;

        public Location(double lat, double lon, double weight) {
            this.lat = lat;
            this.lon = lon;
            this.weight = weight;
        }

        public double getLat() {
            return lat;
        }

        public double getLon() {
            return lon;
        }

        public double getWeight() {
            return weight;
        }

    }
}
