package ru.yandex.crypta.api.rest.resource.portal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiParam;

import ru.yandex.crypta.clients.bigb.BigbIdType;
import ru.yandex.crypta.clients.bigb.ProfileService;
import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.GraphService;
import ru.yandex.crypta.graph.api.service.VultureGraphService;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.proto.Me.Profile;

import static java.util.Map.entry;

@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
@Api(tags = {"portal"})
public class MeResource {

    public static final Map<EIdType, BigbIdType> idTypeToBigbTypeMap = Map.ofEntries(
            entry(EIdType.YANDEXUID, BigbIdType.BIGB_UID),
            entry(EIdType.XUNIQ_GUID, BigbIdType.BIGB_UID),
            entry(EIdType.CRYPTA_ID, BigbIdType.CRYPTA_ID),
            entry(EIdType.PUID, BigbIdType.PUID),
            entry(EIdType.GAID, BigbIdType.GAID),
            entry(EIdType.IDFA, BigbIdType.IDFA),
            entry(EIdType.OAID, BigbIdType.OAID),
            entry(EIdType.MM_DEVICE_ID, BigbIdType.MM_DEVICE_ID),
            entry(EIdType.IFV, BigbIdType.IFV),
            entry(EIdType.DUID, BigbIdType.DUID),
            entry(EIdType.ICOOKIE, BigbIdType.ICOOKIE)
    );

    public static final Set<String> FILTER_BIGB_TYPES = idTypeToBigbTypeMap
            .keySet()
            .stream()
            .map(Soup.CONFIG::name)
            .collect(Collectors.toSet());

    private final GraphService graphService;
    private ProfileService profileService;

    @QueryParam("lang")
    @ApiParam(value = "Language")
    private Language language;

    @Inject
    public MeResource(ProfileService profileService, VultureGraphService graphService) {
        this.graphService = graphService;
        this.profileService = profileService;
    }


    @GET
    @Path("profile")
    public Profile getProfile(
            @QueryParam("uidType") @ApiParam(value = "BigB uid type") @NotNull BigbIdType bigbIdType,
            @QueryParam("uid") @ApiParam(value = "BigB uid value") @NotNull String bigbIdValue
    ) {
        return profileService.get(bigbIdType, bigbIdValue, language);
    }

    @GET
    @Path("identifiers")
    @RolesAllowed({Roles.Portal.PROFILE})
    public Map<String, List<String>> getIdentifiers(
            @QueryParam("idType") @ApiParam(value = "idserv uid type") @NotNull String idType,
            @QueryParam("idValue") @ApiParam(value = "idserv uid value") @NotNull String idValue
    ) {
        // TODO: get directly from BigB answer (source_uniqs keyword 725 or 954)

        GraphId graphId = new GraphId(idValue, idType);
        Graph graph = graphService
                .getById(graphId)
                .orElseThrow(Exceptions::notFound);

        Map<String, List<String>> identifiersMap = new HashMap<>();

        graph.getGraphComponents().forEach(component -> {
            component.getVertices().forEach(vertex -> {
                String vertexIdType = vertex.getIdType();

                if (FILTER_BIGB_TYPES.contains(vertexIdType)) {
                    identifiersMap
                            .computeIfAbsent(vertexIdType, t -> new ArrayList<>())
                            .add(vertex.getIdValue());
                }
            });

            identifiersMap
                    .computeIfAbsent(GraphId.CRYPTA_ID_TYPE, t -> new ArrayList<>())
                    .add(component.getCryptaId());
        });

        return identifiersMap;
    }
}

