package ru.yandex.crypta.api.rest.resource.portal;

import java.util.List;
import java.util.stream.Collectors;

import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.SecurityContext;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import okhttp3.HttpUrl;

import ru.yandex.commune.json.jackson.JodaTimeModule;
import ru.yandex.commune.json.jackson.StringEnumModule;
import ru.yandex.commune.json.jackson.bolts.BoltsModule;
import ru.yandex.crypta.clients.bigb.BigbIdType;
import ru.yandex.crypta.clients.bigb.PublicProfileService;
import ru.yandex.crypta.clients.idm.RoleService;
import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.graph.api.service.PublicGraphService;
import ru.yandex.crypta.idm.Role;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lib.proto.TStaffConfig;
import ru.yandex.crypta.proto.PublicCryptaId;
import ru.yandex.crypta.proto.PublicGraph;
import ru.yandex.crypta.proto.PublicProfile;
import ru.yandex.crypta.proto.PublicTransaction;
import ru.yandex.crypta.service.tx.TxService;
import ru.yandex.inside.staff.v3.StaffApi;
import ru.yandex.inside.staff.v3.model.Email;
import ru.yandex.inside.staff.v3.model.Phone;


@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
@Api(tags = {"portal"})
public class PublicPortalResource {

    private final PublicProfileService publicProfileService;
    private final RoleService roleService;
    private final TxService txService;
    private final PublicGraphService publicGraphService;
    private final StaffApi staffClient;

    @Context
    private SecurityContext securityContext;

    @QueryParam("lang")
    @ApiParam(value = "Language")
    private Language language;

    @Inject
    public PublicPortalResource(
            PublicProfileService publicProfileService,
            TxService txService,
            PublicGraphService publicGraphService,
            RoleService roleService,
            TStaffConfig staffConfig
    ) {
        this.roleService = roleService;
        this.publicProfileService = publicProfileService;
        this.txService = txService;
        this.publicGraphService = publicGraphService;
        this.staffClient = StaffApi.builder()
                .endpoint(new HttpUrl.Builder().scheme("https").host(staffConfig.getUrl()).toString())
                .oauth(staffConfig.getOauthToken())
                .mapper(new ObjectMapper()
                        .registerModule(new BoltsModule())
                        .registerModule(new StringEnumModule())
                        .registerModule(new JodaTimeModule()))
                .build();
    }

    @GET
    @ApiOperation(value = "Get public profile")
    @Path("public_profile")
    public PublicProfile getPublicProfile(
            @QueryParam("uidType") @ApiParam(value = "BigB uid type") @NotNull BigbIdType bigbIdType,
            @QueryParam("uid") @ApiParam(value = "BigB uid value") @NotNull String bigbIdValue
    ) {
        return publicProfileService.get(bigbIdType, bigbIdValue, language);
    }

    @GET
    @ApiOperation(value = "Get public transactions")
    @Path("public_transactions")
    public List<PublicTransaction> getPublicTransactions(
            @QueryParam("uidType") @ApiParam(value = "BigB uid type") @NotNull BigbIdType bigbIdType,
            @QueryParam("uid") @ApiParam(value = "BigB uid value") @NotNull String bigbIdValue
    ) {
        return txService.getPublicTransactions(bigbIdType, bigbIdValue);
    }


    private String getCryptaIdByIdWithDefault(String id) {
        var foundCryptaIds = publicGraphService.getCryptaId(id);
        if (!foundCryptaIds.isEmpty()) {
            return foundCryptaIds.get(0).getCryptaId();
        }
        return "";
    }

    private boolean visitorsIdsAllowed(List<String> visitorIds, List<PublicCryptaId> publicCryptaIds) {
        List<String> visitorPhoneCryptaIds = visitorIds.stream()
                .map(this::getCryptaIdByIdWithDefault)
                .filter(cryptaId -> !cryptaId.isEmpty())
                .collect(Collectors.toList());
        return !visitorPhoneCryptaIds.isEmpty() && visitorPhoneCryptaIds.contains(publicCryptaIds.get(0).getCryptaId());
    }

    @GET
    @ApiOperation(value = "Get public crypta id")
    @Path("public_crypta_id")
    public List<PublicCryptaId> getPublicCryptaId(
            @QueryParam("id") @ApiParam(value = "Personal identificator") @NotNull String id
    ) {
        String login = securityContext.getUserPrincipal().getName();
        List<Role> roles = roleService.getRoles(login);
        var isAdmin = roles.stream().anyMatch(role -> role.getGroup().equals(Roles.ADMIN));
        var requestedCryptaIds = publicGraphService.getCryptaId(id);

        if (isAdmin || login.equals("ndozmorov")) {
            return requestedCryptaIds;
        }

        if (requestedCryptaIds.isEmpty()) {
            return requestedCryptaIds;
        }

        // If a user not admin get their phones, emails and yandex login from staff api and get crypta_id by phones
        var response = staffClient.persons().getO(login);

        List<String> phones;
        List<String> emails;
        String yaLogin = "";

        if (!response.isPresent()) {
            throw Exceptions.notAuthenticated(String.format("Did not found login=%s in staff api", login));
        } else {
            var responceInfo = response.get();

            phones = responceInfo.getPhones().stream().map(Phone::getNumber).collect(Collectors.toList());
            emails = responceInfo.getEmails().stream().map(Email::getAddress).collect(Collectors.toList());
            var optionalAccount = responceInfo.getYandexAccount();
            if (optionalAccount.isPresent()) {
                if (optionalAccount.get().getLoginO().isPresent()) {
                    yaLogin = optionalAccount.get().getLoginO().get();
                }
            }
        }

        // Check if yandex login crypta_id equals requested crypta_id
        String visitorLoginCryptaId = "";
        if (!yaLogin.isEmpty()) {
            var visitorCryptaIds = publicGraphService.getCryptaId(yaLogin);
            if (!visitorCryptaIds.isEmpty()) {
                visitorLoginCryptaId = visitorCryptaIds.get(0).getCryptaId();
            }
        }
        if (!visitorLoginCryptaId.isEmpty() && visitorLoginCryptaId.equals(requestedCryptaIds.get(0).getCryptaId())) {
            return requestedCryptaIds;
        }

        // Check if phones crypta_ids contains requested crypta_id
        if (visitorsIdsAllowed(phones, requestedCryptaIds)) {
            return requestedCryptaIds;
        }

        // Check if emails crypta_ids contains requested crypta_id
        if (visitorsIdsAllowed(emails, requestedCryptaIds)) {
            return requestedCryptaIds;
        }

        throw Exceptions.notAuthenticated(String.format("Requested id=%s does not belong to login=%s", id, login));
    }

    @GET
    @ApiOperation(value = "Get public graph")
    @Path("public_graph")
    public PublicGraph getPublicGraph(
            @QueryParam("crypta_id") @ApiParam(value = "Crypta id") @NotNull String cryptaId
    ) {
        return publicGraphService.getGraph(cryptaId);
    }
}
