package ru.yandex.crypta.api.rest.resource.portal.grab;

import java.nio.charset.StandardCharsets;
import java.util.Optional;
import java.util.UUID;

import javax.annotation.security.RolesAllowed;
import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.SecurityContext;

import io.swagger.annotations.ApiOperation;

import ru.yandex.crypta.clients.pgaas.PostgresClient;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.idm.Roles;

@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class GrabResource {

    private final GrabTable table;

    @Context
    SecurityContext securityContext;

    @Inject
    public GrabResource(PostgresClient postgresClient) {
        this.table = new GrabTable(postgresClient.getJooqConfiguration());
    }

    private static void validateGrabId(GrabId grabId) {
        Optional<String> grabIdDecoded = Base58.decodeCheck(grabId.getValue());
        if (!grabIdDecoded.isPresent()) {
            throw Exceptions.illegal("Invalid grab id");
        }
    }

    @POST
    @ApiOperation(value = "Insert or update grab")
    @RolesAllowed({Roles.Portal.GRAB})
    public Grab putGrab(@NotNull Grab grab) {
        validateGrabId(grab.getGrabId());
        table.upsertQuery(grab).execute();
        return grab;
    }

    @GET
    @ApiOperation(value = "Create new grab id")
    public GrabId newGrabId() {
        String login = securityContext.getUserPrincipal().getName();
        UUID uuid = UUID.randomUUID();
        String raw = login + "|" + uuid.toString();
        // we obfuscate it a bit
        return new GrabId(Base58.encodeCheck(raw.getBytes(StandardCharsets.US_ASCII)));
    }

    @GET
    @Path("{grab_id}")
    @ApiOperation(value = "Get grab by id", nickname = "grab-getGrab")
    @RolesAllowed({Roles.Portal.GRAB, Roles.Portal.PROFILE})
    public Grab getGrab(@PathParam("grab_id") @NotNull String grabId) {
        validateGrabId(new GrabId(grabId));
        Optional<Grab> grab = table.selectByIdQuery(grabId).fetchOptionalInto(Grab.class);
        if (grab.isPresent()) {
            return grab.get();
        } else {
            throw Exceptions.notFound();
        }
    }

}
