package ru.yandex.crypta.api.rest.resource.search;

import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiParam;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.search.SearchService;
import ru.yandex.crypta.search.proto.Search.TResultResponse;
import ru.yandex.crypta.search.proto.Service;

@Api(tags = {"search"})
@Path("search")
@Produces(JsonUtf8.MEDIA_TYPE)
@Consumes(JsonUtf8.MEDIA_TYPE)
public class SearchResource {

    private final SearchService search;

    @QueryParam("lang")
    @ApiParam(value = "Language")
    private Language language;

    @Inject
    public SearchResource(SearchService search) {
        this.search = search;
    }

    @GET
    public TResultResponse search(
            @QueryParam("query") @ApiParam(value = "Query") @NotNull String query,
            @QueryParam("matcher") @ApiParam(value = "Matcher") String matcher,
            @QueryParam("nocache") @ApiParam(value = "No Cache") boolean nocache,
            @Context HttpHeaders headers

    ) {
        var request = Service.TSearchRequest.newBuilder()
                .setQuery(query)
                .setNoCache(nocache);
        if (Objects.nonNull(matcher)) {
            request.putArguments("matcher", matcher);
        }

        if (language == null) {
            language = extractLocaleLanguage(headers);
        }

        return search.search(request.build(), language);
    }

    private Language extractLocaleLanguage(HttpHeaders headers) {
        List<Locale> languages = headers.getAcceptableLanguages();
        if (languages != null) {
            List<String> localeLanguages = languages
                    .stream()
                    .map(Locale::getLanguage)
                    .filter(l -> !l.isEmpty())
                    .map(String::toUpperCase)
                    .collect(Collectors.toList());

            for (String localeLanguage : localeLanguages) {
                if (Language.ALL_SUPPORTED.containsKey(localeLanguage)) {
                    return Language.ALL_SUPPORTED.get(localeLanguage);
                }
            }
        }
        return null;
    }

    @GET
    @Path("examples")
    public Service.TExamplesResponse examples() {
        return Service.TExamplesResponse.newBuilder().addAllExamples(search.examples()).build();
    }
}
