package ru.yandex.crypta.api.rest.resource.task;

import java.util.List;
import java.util.Map;

import javax.inject.Inject;
import javax.inject.Named;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.common.ws.auth.AuthNotRequired;
import ru.yandex.crypta.common.ws.jersey.JsonUtf8;
import ru.yandex.crypta.service.task.JugglerTaskStatusReporter;
import ru.yandex.crypta.service.task.SolomonTaskStatusReporter;
import ru.yandex.crypta.service.task.TaskStatusService;
import ru.yandex.crypta.service.task.model.CryptaTaskStatusChangedEvent;

@Path("task")
@Produces(JsonUtf8.MEDIA_TYPE)
@Api(tags = "task")
public class TaskStatusReportResource {

    private static final Logger LOG = LoggerFactory.getLogger(TaskStatusReportResource.class);

    private final TaskStatusService stepTaskStatusService;
    private final TaskStatusService reactorTaskStatusService;
    private final SolomonTaskStatusReporter solomonReporter;
    private final JugglerTaskStatusReporter jugglerReporter;

    @Inject
    public TaskStatusReportResource(
            @Named("step") TaskStatusService stepTaskStatusService,
            @Named("reactor") TaskStatusService reactorTaskStatusService,
            SolomonTaskStatusReporter solomonReporter, JugglerTaskStatusReporter jugglerReporter) {
        this.stepTaskStatusService = stepTaskStatusService;
        this.reactorTaskStatusService = reactorTaskStatusService;
        this.solomonReporter = solomonReporter;
        this.jugglerReporter = jugglerReporter;
    }

    @Path("status")
    @POST
    @AuthNotRequired
    @ApiOperation("Report new task status")
    @Produces(MediaType.APPLICATION_JSON)
    public CryptaTaskStatusChangedEvent reportNewTaskStatus(
            @QueryParam("taskType") String taskType,
            @QueryParam("taskStatus") String taskStatus,
            @QueryParam("parentTaskInstances") List<String> parentInstanceIds,
            @QueryParam("environment") String environment,
            Map<String, String> taskParams
    )
    {
        CryptaTaskStatusChangedEvent changedEvent = new CryptaTaskStatusChangedEvent(
                taskType, taskStatus, environment, taskParams
        );

        try {
            stepTaskStatusService.reportNewTaskStatus(changedEvent);
        } catch (Exception e) {
            LOG.error("{}: Can't report task status to Step: {}", changedEvent.getTaskInstanceId(), taskType, e);
        }

        try {
            reactorTaskStatusService.reportNewTaskStatus(changedEvent);
        } catch (Exception e) {
            LOG.error("{}: Can't report task status to Reactor: {}", changedEvent.getTaskInstanceId(), taskType, e);
        }

        try {
            solomonReporter.reportTaskStatus(taskType, taskStatus, environment, taskParams);
        } catch (Exception e) {
            LOG.error("{}: Can't report task status to Solomon: {}", changedEvent.getTaskInstanceId(), taskType, e);
        }

        try {
            jugglerReporter.reportTaskStatus(taskType, taskStatus, environment, taskParams);
        } catch (Exception e) {
            LOG.error("{}: Can't report task status to Juggler: {}", changedEvent.getTaskInstanceId(), taskType, e);
        }

        return changedEvent;
    }

    @Path("status/{taskInstanceId}")
    @PUT
    @AuthNotRequired
    @ApiOperation("Update task status")
    public void reportExistingTaskStatus(
            @PathParam("taskInstanceId") String taskInstanceId,
            @QueryParam("taskType") String taskType,
            @QueryParam("taskStatus") String taskStatus,
            @QueryParam("environment") String environment,
            Map<String, String> taskParams
    )
    {
        stepTaskStatusService.reportExistingTaskStatus(
                taskInstanceId, taskType, taskStatus, environment, taskParams
        );

        try {
            reactorTaskStatusService.reportExistingTaskStatus(
                    taskInstanceId, taskType, taskStatus, environment, taskParams
            );
        } catch (Exception e) {
            LOG.error("Can't report existing task status to Reactor: {}", taskType, e);
        }
    }

}
