package ru.yandex.crypta.audience;

import java.util.ArrayList;
import java.util.List;
import java.util.NoSuchElementException;

import javax.inject.Inject;
import javax.validation.constraints.NotNull;

import com.google.protobuf.InvalidProtocolBufferException;

import ru.yandex.crypta.audience.proto.TSegmentState;
import ru.yandex.crypta.audience.proto.TUserDataStats;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lib.yt.YtReadingUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

public class DefaultAudienceService implements AudienceService {

    private final Yt yt;

    @Inject
    public DefaultAudienceService(YtService yt) {
        this.yt = yt.getHahn();
    }

    @Override
    public TUserDataStats getStats(String type, String id) {
        YPath table = getExisting(getStatsTablePath());
        YTreeMapNode row = YtReadingUtils
                .lookupSingleRow(yt, table, segmentKey(type, Integer.valueOf(id)), YTableEntryTypes.YSON)
                .orElseThrow(Exceptions::notFound);
        return getUserDataStats(row);
    }

    @Override
    public TSegmentState getState(String type, String id) {
        YPath table = getExisting(getStateTablePath(type));
        return YtReadingUtils
                .lookupSingleRow(yt, table, segmentKey("audience", Integer.valueOf(id)),
                        YTableEntryTypes.proto(TSegmentState.class))
                .orElseThrow(Exceptions::notFound);
    }

    @Override
    public List<String> getSegments() {
        List<String> results = new ArrayList<>();
        yt.tables().selectRows(
                String.format("SegmentID FROM [%s] ORDER BY Hash LIMIT 10", getStatsTablePath()),
                YTableEntryTypes.YSON,
                (each) -> {
                    results.add(String.valueOf(
                            each.get("SegmentID")
                                    .orElseThrow(() -> new NoSuchElementException("Missed segment ID")).intValue()
                    ));
                }
        );
        return results;
    }

    private YPath getStateTablePath(String type) {
        return YPath.simple("//home/crypta/production/audience/dynamic/states").child(type);
    }

    private YPath getStatsTablePath() {
        return YPath.simple("//home/crypta/production/audience/Stats");
    }

    private YPath getExisting(YPath path) {
        if (!yt.cypress().exists(path)) {
            throw Exceptions.notFound();
        }
        return path;
    }

    private TUserDataStats getUserDataStats(YTreeMapNode row) {
        try {
            return TUserDataStats.parseFrom(row.getOrThrow("Stats").bytesValue());
        } catch (InvalidProtocolBufferException e) {
            throw Exceptions.unavailable();
        }
    }

    private YTreeMapNode segmentKey(String type, @NotNull Integer id) {
        return YTree.mapBuilder()
                .key("SegmentID")
                .value(id)
                .key("SegmentType")
                .value(type)
                .buildMap();
    }

}
