#include <library/cpp/dot_product/dot_product.h>
#include <crypta/audience/lib/native/audience_geo.h>
#include <util/generic/hash_set.h>

using namespace NYT;
using namespace NAudience;
using namespace NLab;

void TMatchWithUserData::Do(TTableReader<TPrunedUserData>* input, TTableWriter<TUserDataWithSegments>* output) {
    if (input->GetTableIndex() != 0) {
        return;
    }

    TUserDataWithSegments userDataWithSegments;
    userDataWithSegments.MutableUserData()->CopyFrom(input->MoveRow().GetUserData());
    userDataWithSegments.MutableUserData()->ClearAffinities();
    TUserDataWithSegments::TSegments segments;
    for (; input->IsValid(); input->Next()) {
        if (input->GetTableIndex() != 0) {
            const auto row = input->MoveRow();
            segments.AddSegmentID(row.GetSegmentID());
        }
    }
    userDataWithSegments.MutableSegments()->CopyFrom(segments);
    if (segments.GetSegmentID().size()) {
        output->AddRow(userDataWithSegments);
    }
}

void TCountMapper::Start(TTableWriter<TCountedSegmentID>*) {
    Counts.clear();
}

void TCountMapper::Do(TTableReader<TCountedSegmentID>* input, TTableWriter<TCountedSegmentID>*) {
    for (; input->IsValid(); input->Next()) {
        Counts[input->MoveRow().GetSegmentID()] += 1;
    }
}

void TCountMapper::Finish(TTableWriter<TCountedSegmentID>* output) {
    for (const auto& each : Counts) {
        TCountedSegmentID count;
        count.SetSegmentID(each.first);
        count.SetCount(each.second);
        output->AddRow(count);
    }
}

void TCountReducer::Do(TTableReader<TCountedSegmentID>* input, TTableWriter<TCountedSegmentID>* output) {
    TCountedSegmentID out;
    out.SetSegmentID(input->GetRow().GetSegmentID());
    out.SetGroupID(ToString(out.GetSegmentID()));
    ui64 totalCount = 0;
    for (; input->IsValid(); input->Next()) {
        ui64 count = input->MoveRow().GetCount();
        if (!count) {
            count = 1;
        }
        totalCount += count;
    }
    out.SetCount(totalCount);
    output->AddRow(out);
}

void TTransformCountsToStats::Do(TTableReader<TCountedSegmentID>* input, TTableWriter<NLab::TUserDataStats>* output) {
    for (; input->IsValid(); input->Next()) {
        auto row = input->MoveRow();
        auto count = row.GetCount();
        NLab::TUserDataStats out;
        out.SetGroupID(ToString(row.GetSegmentID()));

        auto counts = out.MutableCounts();
        counts->SetUniqYuid(count);
        counts->SetUniqIdValue(count);
        output->AddRow(out);
    }
}

void TMergeStatsWithCounts::Do(TTableReader<NLab::TUserDataStats>* input, TTableWriter<NLab::TUserDataStats>* output) {
    NLab::TUserDataStats out;
    ui64 count = 0;
    for (; input->IsValid(); input->Next()) {
        auto row = input->MoveRow();
        out.SetGroupID(row.GetGroupID());
        auto tmpCount = row.GetCounts().GetUniqIdValue();
        if (tmpCount) {
            count = tmpCount;
        } else if (row.GetCounts().GetWithData()) {
            out = row;
        }
    }
    if (count) {
        auto counts = out.MutableCounts();
        counts->SetUniqYuid(count);
        counts->SetUniqIdValue(count);
    }
    output->AddRow(out);
}

void TExtractGeoStorageOutput::Start(TTableWriter<TStorageOutput>*) {
    OutputSegments.clear();
    for (const auto& segment : State->GetOutputSegments().GetSegmentID()) {
        OutputSegments.insert(segment);
    }
}

void TExtractGeoStorageOutput::Do(TTableReader<TInput>* input, TTableWriter<TStorageOutput>* output) {
    for (; input->IsValid(); input->Next()) {
        auto in = input->MoveRow();
        TStorageOutput out;
        auto segmentID = in.GetSegmentID();
        if (!OutputSegments.count(segmentID)) {
            continue;
        }

        out.SetYandexuid(ToString(in.GetUserID()));
        out.SetExternalID(ToString(segmentID));
        out.SetTimestamp(State->GetTimestamp());
        output->AddRow(out);
    }
}

void TSplitUserDataToSegmentStats::Do(TTableReader<TUserDataWithSegments>* input, TTableWriter<TUserDataStats>* output) {
    TUserDataStats userDataStats;

    for (ui64 rowCount = 0; input->IsValid(); input->Next(), ++rowCount) {
        const auto& row = input->GetRow();
        const auto& userData = row.GetUserData();
        Aggregator.ConvertUserDataToUserDataStats(userDataStats, userData);

        if (userData.GetGroupID() && State->GetFlags().GetDuplicateWithoutGroupID()) {
            userDataStats.ClearGroupID();
            Aggregator.UpdateWith(userDataStats);
        }

        for (const auto& groupID : row.GetSegments().GetSegmentID()) {
            Aggregator.FillMetaFromOptions(userDataStats, userData);
            userDataStats.SetGroupID(ToString(groupID));
            if (AggregateSegmentsFlag) {
                Aggregator.UpdateWith(userDataStats);
            } else {
                output->AddRow(userDataStats);
            }
        }

        Aggregator.EmitIfLarge(output);
    }
}

void TSplitUserDataToSegmentStats::Start(TTableWriter<TUserDataStats>* output) {
    Aggregator.Start(output, State);
}

void TSplitUserDataToSegmentStats::Finish(TTableWriter<TUserDataStats>* output) {
    Aggregator.Finish(output);
}
