#!/usr/bin/env python

from flask import Flask
import socket
import logging
from time import time
import os

log = logging.getLogger(__name__)
log.setLevel(logging.ERROR)


app = Flask(
    __name__,
)


# Watchdog restarts app container sendig SIGKILL to ppid to force chain restart
# so monitoring process must be single-processed and spawned from the
# initial script itself
class WatchDog(object):
    WATCHDOG_TIMEOUT = 60 * 3
    WATCHDOG_FILE = "/tmp/watchdog"

    def __init__(self, watchdog_file=WATCHDOG_FILE, timeout=WATCHDOG_TIMEOUT):
        self.file = watchdog_file
        self.timeout = timeout
        self.update()

    def update(self):
        with open(self.file, "w") as outfile:
            outfile.write(str(time()))

    def good(self):
        self.update()

    def bad(self):
        if time() - self.get_last_ok() > self.timeout:
            self.restart()

    def get_last_ok(self):
        try:
            return float(open(self.file).read())
        except:
            return time() - self.timeout - 1    # restart if smth goes wrong

    def restart(self):
        os.kill(os.getppid(), 9)


def check_zk_role(zk_watchdog):

    ZK_CLIENT_PORT = 2181

    def is_good_reply(reply):
        return reply[-1] == 200

    def parse_zk_reply(data):
        result = []
        for x in data.split('\n'):
            try:
                n, v = x.split()
            except ValueError:
                continue
            if n == 'Mode:' and v in ['follower', 'leader']:
                return 'OK', 200

        return 'CRIT', 500

    s = socket.socket(socket.AF_INET, socket.SOCK_STREAM, socket.IPPROTO_TCP)
    try:
        s.connect(('127.0.0.1', ZK_CLIENT_PORT))
    except (socket.timeout, socket.error):
        zk_watchdog.bad()
        return 'CRIT', 500

    s.send('stat')
    data = s.recv(65535)
    s.shutdown(2)
    s.close()

    reply = parse_zk_reply(data)
    if is_good_reply(reply):
        zk_watchdog.good()
    else:
        zk_watchdog.bad()

    return reply


@app.route('/ping')
def ping():
    return check_zk_role(app.watchdog)


if __name__ == '__main__':
    app.watchdog = WatchDog()
    app.run(
        host  = '::',
        port  = 80,
    )
