from datetime import timedelta

import yt.wrapper as yt

from crypta.lib.python import crypta_env
from crypta.lib.python.bt.conf import (
    conf,
    resource_conf,
)
from crypta.lib.python.data_size import DataSize
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine import deploy
from crypta.lib.python.spine.consts import (
    dc as dc_consts,
    environment,
)
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler import (
    consts,
    juggler_check_generator,
)
from crypta.lib.python.spine.sandbox import sandbox_scheduler
from crypta.lib.python.spine.solomon import (
    solomon_alert_utils,
    solomon_check_generator,
)
from crypta.lib.python.spine.yt import yt_config_registry
from crypta.lib.python.spine.yt.yt_latency_metric import YtLatencyMetric
from crypta.lib.python.spine.yt.yt_size_metric import YtSizeMetric
from sandbox.projects.crypta import audience


CRYPTADEVNETS = "_CRYPTADEVNETS_"
CRYPTANETS = "_CRYPTANETS_"


class Task(object):
    def __init__(self, task, envs, crit_time, kill_timeout, schedule_interval=None, task_args=None, schedule_daily_start_time=None, retry_interval=None):
        self.task = task
        self.envs = envs
        self.crit_time = crit_time
        self.kill_timeout = kill_timeout
        self.schedule_interval = schedule_interval
        self.task_args = task_args
        self.schedule_daily_start_time = schedule_daily_start_time
        self.retry_interval = retry_interval


def get_registry():
    juggler = juggler_check_generator.CryptaYtCheckGenerator(
        host="crypta-audience",
        child_group_type=consts.GroupType.host,
        child_group="crypta-audience-solomon",
        tags=["crypta-audience"],
    )
    add_yt_metrics(juggler)
    add_solomon_alerts(juggler)
    add_sandbox(juggler)
    add_audience_service(juggler)
    return juggler


def add_sandbox(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["AUDIENCE"])
    all_envs = (environment.STABLE, environment.TESTING)
    prod_envs = (environment.STABLE,)

    for task in (
        Task(
            task="crypta.audience.lib.storage.DistributeCryptaidorSegmentsForFull",
            envs=prod_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_daily_start_time="2020-09-04T13:00:00Z",
            kill_timeout=timedelta(hours=3),
            retry_interval=timedelta(hours=1),
        ),
        Task(
            task="crypta.audience.lib.matching.Prepare",
            envs=all_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_daily_start_time="2020-02-13T07:00:53Z",
            kill_timeout=timedelta(hours=3),
            retry_interval=timedelta(hours=2),
            task_args={
                "proxy": "hahn.yt.yandex.net",
                "day": "yesterday",
            },
        ),
        Task(
            task="crypta.audience.lib.matching.PrepareCryptaIds",
            envs=prod_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_daily_start_time="2022-02-13T00:00:00Z",
            kill_timeout=timedelta(hours=3),
            retry_interval=timedelta(hours=2),
            task_args={
                "proxy": "hahn.yt.yandex.net",
                "day": "yesterday",
            },
        ),
        Task(
            task="crypta.audience.lib.tasks.audience.precompute_communalities.PrecomputeCommunalities",
            envs=all_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_interval=timedelta(hours=24),
            kill_timeout=timedelta(hours=1),
            task_args={
                "proxy": "hahn.yt.yandex.net",
                "day": "today",
            },
        ),
        Task(
            task="crypta.audience.lib.tasks.audience.geo.CreatePrunedUserData",
            envs=prod_envs,
            crit_time=timedelta(days=1, hours=12),
            schedule_interval=timedelta(hours=2),
            kill_timeout=timedelta(hours=4),
        ),
        Task(
            task="crypta.audience.lib.storage.UpdateFullYandexuid",
            envs=all_envs,
            crit_time=timedelta(hours=24),
            schedule_interval=timedelta(hours=2, minutes=30),
            kill_timeout=timedelta(hours=16),
        ),
        Task(
            task="crypta.audience.lib.storage.UpdateFullCryptaId",
            envs=all_envs,
            crit_time=timedelta(hours=12),
            schedule_interval=timedelta(minutes=30),
            kill_timeout=timedelta(hours=10),
        ),
        Task(
            task="crypta.audience.lib.storage.UpdateFullPuid",
            envs=all_envs,
            crit_time=timedelta(hours=12),
            schedule_interval=timedelta(minutes=30),
            kill_timeout=timedelta(hours=10),
        ),
        Task(
            task="crypta.audience.lib.storage.UpdateFullDevices",
            envs=all_envs,
            crit_time=timedelta(hours=12),
            schedule_interval=timedelta(minutes=30),
            kill_timeout=timedelta(hours=10),
        ),
        Task(
            task="crypta.audience.lib.storage.UpdateLimited",
            envs=all_envs,
            crit_time=timedelta(hours=12),
            schedule_interval=timedelta(minutes=5),
            kill_timeout=timedelta(hours=6),
        ),
        Task(
            task="crypta.audience.lib.storage.UploadSegmentPrioritiesToSandbox",
            envs=all_envs,
            crit_time=timedelta(hours=12),
            schedule_interval=timedelta(minutes=30),
            kill_timeout=timedelta(hours=1),
        ),
        Task(
            task="crypta.audience.lib.storage.DumpBigbSample",
            envs=prod_envs,
            crit_time=timedelta(hours=12),
            schedule_interval=timedelta(hours=1),
            kill_timeout=timedelta(hours=1),
        ),
        Task(
            task="crypta.audience.lib.storage.CheckDesync",
            envs=prod_envs,
            crit_time=timedelta(days=2),
            schedule_interval=timedelta(days=1),
            kill_timeout=timedelta(hours=12),
        ),
    ):
        extra_params = {"task_name": task.task}
        if task.task_args:
            extra_params["task_args"] = task.task_args

        for env in task.envs:
            scheduler = sandbox.create_scheduler(
                audience.CryptaAudience,
                schedule_interval=task.schedule_interval,
                schedule_daily_start_time=task.schedule_daily_start_time,
                kill_timeout=task.kill_timeout,
                extra_params=extra_params,
                env=env,
                retry_interval=task.retry_interval,
            )
            if env == environment.STABLE:
                scheduler.check(task.crit_time, juggler_service=audience.CryptaAudience.get_juggler_service_for_task_name(task.task)).add_yt_dependencies(YtProxy.Group.offline)


def add_yt_metrics(juggler):
    yt_registry = yt_config_registry.CryptaYtConfigRegistry(juggler)

    confs = resource_conf.find("/crypta/audience")
    paths = conf.use_configs(conf.DictLocator(confs), [])["paths"]

    def get_subpath(path):
        return path[len(paths.new_root)+1:]

    for path in (
        paths.storage.full_yandexuid,
        paths.storage.full_crypta_id,
        paths.storage.full_devices,
    ):
        YtSizeMetric(yt_registry, get_subpath(path))

    for path, threshold in (
        (paths.audience.input, timedelta(hours=24)),
        (paths.audience.output, timedelta(hours=48)),
        (paths.audience.batches, timedelta(hours=24)),
        (yt.ypath_join(paths.lookalike.input, "audience"), timedelta(hours=24)),
        (yt.ypath_join(paths.lookalike.output, "audience"), timedelta(hours=48)),
        (paths.lookalike.segments.batches, timedelta(hours=24)),
        (paths.storage.for_full, timedelta(hours=12)),
    ):
        YtLatencyMetric(yt_registry, get_subpath(path)).add_table_latency_alert(threshold)

    for path, threshold in (
        (paths.audience.input_batch, timedelta(hours=24)),
        (paths.audience.output_batch, timedelta(hours=24)),
        (paths.lookalike.segments.waiting, timedelta(hours=24)),
    ):
        YtLatencyMetric(yt_registry, get_subpath(path)).add_latency_alert(threshold)


def add_solomon_alerts(juggler):
    solomon = solomon_check_generator.SolomonCheckGenerator(
        juggler,
        solomon_alert_utils.AlertCreator(
            project_id="crypta_audience",
            selectors={"project": "crypta_audience", "cluster": "production"},
        ),
    )
    notification_timedelta = timedelta(hours=3)

    solomon.get_sensor({"sensor": "input_size", "service": "metrics_audience"}).create_threshold_check(
        aggregation=alert_pb2.MAX,
        predicate=alert_pb2.GT,
        threshold=25000,
        period=notification_timedelta,
        juggler_service="CryptaAudience_queues_regular_input",
        description="Input size (regular): {{ pointValue }}",
    )
    solomon.get_sensor({"sensor": "batches_size", "service": "metrics_audience"}).create_threshold_check(
        aggregation=alert_pb2.MAX,
        predicate=alert_pb2.GT,
        threshold=40000,
        period=notification_timedelta,
        juggler_service="CryptaAudience_queues_regular_batches",
        description="In batches (regular): {{ pointValue }}",
    )
    solomon.get_sensor({"sensor": "output_size", "service": "metrics_audience"}).create_threshold_check(
        aggregation=alert_pb2.MAX,
        predicate=alert_pb2.GT,
        threshold=20000,
        period=notification_timedelta,
        juggler_service="CryptaAudience_queues_regular_output",
        description="Output size (regular): {{ pointValue }}",
    )
    solomon.get_sensor({"sensor": "input_size", "service": "metrics_lookalike"}).create_threshold_check(
        aggregation=alert_pb2.MAX,
        predicate=alert_pb2.GT,
        threshold=5000,
        period=notification_timedelta,
        juggler_service="CryptaAudience_queues_lookalike_input",
        description="Input size (lookalike): {{ pointValue }}",
    )
    solomon.get_sensor({"sensor": "output_size", "service": "metrics_lookalike"}).create_threshold_check(
        aggregation=alert_pb2.MAX,
        predicate=alert_pb2.GT,
        threshold=5000,
        period=notification_timedelta,
        juggler_service="CryptaAudience_queues_lookalike_output",
        description="Output size (lookalike): {{ pointValue }}",
    )
    solomon.get_sensor({"sensor": "online_workers", "service": "metrics_workflow"}).create_threshold_check(
        aggregation=alert_pb2.MAX,
        predicate=alert_pb2.LT,
        threshold=55,
        period=notification_timedelta,
        juggler_service="CryptaAudience_workers_online",
        description="Online workers: {{ pointValue }}",
    )
    solomon.get_sensor({"sensor": "userdata_age", "service": "metrics_audience"}).create_threshold_check(
        aggregation=alert_pb2.MAX,
        predicate=alert_pb2.GT,
        threshold=3000000,
        period=notification_timedelta,
        juggler_service="CryptaAudience_userdata_age",
        description="Userdata age: {{ pointValue }}",
    )

    solomon.create_expression_alert_check(
        service="bigb-audience-segments-consistency",
        period=timedelta(days=2),
        program="""
        let bigb = group_by_time(1d, "avg", {
            project="buzzard", cluster="production", service="push_table_stats",
            experiment="default", sensor="counts/kw_557", type="-", profileAge="-", profileSize="-"
        });

        let state = group_lines("sum",
            group_by_time(1d, "avg", {
                project="crypta_stats", cluster="production", service="yt_sizes", metric="row_count",
                sensor="audience/segments/PrivateFull*", yt_proxy="hahn.yt.yandex.net"
            })
        );

        let ratio = avg(bigb / state);
        let threshold = 0.90;

        alarm_if(ratio < threshold);
        """,
        description="bigb/state: {{ expression.ratio }} < {{ expression.threshold }}",
    )

    solomon.create_expression_alert_check(
        service="check-desync",
        period=timedelta(days=2),
        program="""
        let sample_yuids = group_by_time(1d, "avg", {
            project="crypta_audience", cluster="production", service="metrics_audience", sensor="sample_yuids"
        });

        let desynced_yuids = group_by_time(1d, "avg", {
            project="crypta_audience", cluster="production", service="metrics_audience", sensor="desynced_yuids"
        });

        let ratio = avg(desynced_yuids / sample_yuids);
        let threshold = 0.002;

        alarm_if(ratio > threshold);
        """,
        description="desynced yuids ratio: {{ expression.ratio }} > {{ expression.threshold }}",
    )


def add_audience_service(base_juggler_check_generator):
    deploy_gen = base_juggler_check_generator.add_subregistry(deploy.CryptaDeployStageGenerator())

    for env, stage_id, pods, network, yt_pool in (
        (environment.TESTING, "crypta-audience-testing", 3, CRYPTADEVNETS, "crypta-tests"),
        (environment.PRODUCTION, "crypta-audience-production", 7, CRYPTANETS, "crypta_audience"),
    ):
        stage = deploy_gen.stage(stage_id)
        env_vars = crypta_env.VARS_BY_ENV[env]
        zk_hosts = ",".join("zoo{}.{}.crypta.yandex.net".format(i, env) for i in range(1, 6))

        stage.multi_cluster_deploy_unit(
            pods_per_cluster={dc_consts.SAS: pods},
            cpu_ms=2000,
            ram=DataSize(gb=20),
            storage=deploy.HDD(
                capacity=DataSize(gb=20),
                bandwidth_guarantee=DataSize(mb=30),
                bandwidth_limit=DataSize(mb=60),
            ),
            project_network=network,
            network_bandwidth=DataSize(mb=10),
            id_="main",
        ).deployment_strategy(
            max_unavailable=pods,
        ).box(
            docker_image="crypta/crypta-audience-worker",
        ).docker_release_rule(
            environment.STABLE if env == environment.PRODUCTION else env,
        ).literal_env(
            crypta_env.LiteralEnv("TZ", "Europe/Moscow")
        ).literal_env(
            crypta_env.Production.siberia_host
        ).literal_env(
            crypta_env.LiteralEnv("ZK_HOSTS", zk_hosts)
        ).literal_env(
            crypta_env.LiteralEnv("QLOUD_HTTP_PORT", "80")
        ).literal_env(
            crypta_env.Production.siberia_tvm_id
        ).literal_env(
            crypta_env.LiteralEnv("YT_POOL", yt_pool)
        ).literal_env(
            env_vars.crypta_environment
        ).literal_env(
            crypta_env.LiteralEnv("WORKERS_PER_TAG", "--worker hahn=12")
        ).secret_env(
            crypta_env.crypta_api_oauth
        ).secret_env(
            env_vars.yt_token
        ).secret_env(
            env_vars.yql_token
        ).secret_env(
            env_vars.crypta_audience_tvm_secret
        ).workload().add_coredumps()
