import datetime

from yt.wrapper import ypath

from crypta.buchhalter.services.main.lib.adfox_calc.config_pb2 import TConfig
from crypta.lib.python.yql import yql_helpers
from crypta.lib.python.yt import path_utils
from crypta.lib.python.yt import yt_helpers
from crypta.lib.python.yt.processed_tables_tracker import ProcessedTablesTracker


def add_args_subparser(subparsers):
    parser = subparsers.add_parser("adfox_calc", help="Calculate adfox stats")
    parser.set_defaults(function=calc_stats)
    parser.set_defaults(config_proto_cls=TConfig)


def calc_stats(config, logger):
    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)
    tracker = ProcessedTablesTracker(config.AdfoxEventLog)

    while True:
        with yt_client.Transaction() as tx:
            unprocessed_table_paths = tracker.get_unprocessed_tables(yt_client, 1)
            logger.info("Unprocessed tables %s", unprocessed_table_paths)

            if not unprocessed_table_paths:
                logger.info("No tables to process")
                break

            src_table = unprocessed_table_paths[0]
            logger.info("Processing table %s", src_table)
            run_query(src_table, config, yt_client, tx, logger)

            tracker.add_processed_tables(yt_client, [src_table])


def run_query(src_table, config, yt_client, tx, logger):
    date = path_utils.get_basename(src_table)

    def get_table_name(dir):
        return ypath.ypath_join(dir, date)

    paid_per_segment_stats_table = get_table_name(config.PaidPerSegmentStatsDir)
    paid_per_dmp_stats_table = get_table_name(config.PaidPerDmpStatsDir)
    paid_total_stats_table = get_table_name(config.PaidTotalStatsDir)

    audience_per_segment_stats_table = get_table_name(config.AudiencePerSegmentStatsDir)
    audience_per_owner_stats_table = get_table_name(config.AudiencePerOwnerStatsDir)
    audience_top_owners_stats_table = get_table_name(config.AudienceTopOwnersStatsDir)
    audience_per_group_stats_table = get_table_name(config.AudiencePerGroupStatsDir)
    audience_total_stats_table = get_table_name(config.AudienceTotalStatsDir)

    params = {
        "adfox_event_log_table": src_table,
        "adfox_clients_table": config.AdfoxClientsTable,
        "dmp_tariff_data_table": config.DmpTariffDataTable,
        "audience_segment_owners_table": config.AudienceSegmentOwnersTable,
        "groups_dir": config.GroupsDir,

        "paid_per_segment_stats_table": paid_per_segment_stats_table,
        "paid_per_dmp_stats_table": paid_per_dmp_stats_table,
        "paid_total_stats_table": paid_total_stats_table,

        "audience_per_segment_stats_table": audience_per_segment_stats_table,
        "audience_per_owner_stats_table": audience_per_owner_stats_table,
        "audience_top_owners_stats_table": audience_top_owners_stats_table,
        "audience_per_group_stats_table": audience_per_group_stats_table,
        "audience_total_stats_table": audience_total_stats_table,

        "crypta_identifier_udf_url": config.CryptaIdentifierUdfUrl,
        "top_owners_threshold": config.TopOwnersThreshold,
    }

    yql_helpers.run_query("/query/calc_adfox_stats.yql", config.Yt, params, logger, tx=tx)

    ttl_timedelta = datetime.timedelta(days=config.OutputTtlDays)
    for path in [
        paid_per_segment_stats_table,
        paid_per_dmp_stats_table,
        paid_total_stats_table,
        audience_per_segment_stats_table,
        audience_per_owner_stats_table,
        audience_top_owners_stats_table,
        audience_per_group_stats_table,
        audience_total_stats_table,
    ]:
        yt_helpers.set_ttl_by_table_name(path, ttl_timedelta, yt_client=yt_client)
