import datetime

from library.python.protobuf.json import proto2json
import yt.wrapper as yt
from yt.wrapper import ypath

from crypta.buchhalter.services.main.lib.audience_dmp_report_calc.config_pb2 import TConfig
from crypta.buchhalter.services.main.lib.common import (
    join_acl,
    report_generator,
    stats,
)
from crypta.buchhalter.services.main.lib.common.helpers import (
    indexes,
    month_grouper,
    tables_helpers,
)
from crypta.buchhalter.services.main.lib.common.proto.segment_dmp_id_pb2 import TSegmentDmpId
from crypta.dmp.common.upload_to_audience import segment_name
from crypta.lib.native.yt.processed_tables_tracker.proto.tracked_source_pb2 import TTrackedSource
import crypta.lib.python.audience.client as audience
from crypta.lib.python.yql import yql_helpers
from crypta.lib.python.yt import schema_utils
from crypta.lib.python.yt import yt_helpers
from crypta.lib.python.yt.processed_tables_tracker import ProcessedTablesTracker


DAILY_NAME_FORMAT = "%Y-%m-%d"
DMP_SEGMENT_SOURCE = "yndx-robot-crypta-dmp"
JSON_FORMAT = yt.JsonFormat(attributes={"encode_utf8": False})
PAYMENT_MODEL = {
    542: "CPC",
    1542: "CPM",
}


def add_args_subparser(subparsers):
    parser = subparsers.add_parser("audience_dmp_report_calc", help="Calc dmp per segment-login report")
    parser.set_defaults(function=run)
    parser.set_defaults(config_proto_cls=TConfig)


def run(config, logger):
    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)
    audience_client = audience.PublicApiAudienceClient.create_from_proto(config.Audience, logger)

    segment_owners = indexes.get_audience_logins(yt_client, config.DmpIndexTable)
    segment_owner_to_dirname = {x: x for x in segment_owners}

    daily_ttl = datetime.timedelta(days=config.DailyReportTtlDays)
    daily_report_generator = report_generator.ReportGenerator(config.DailyReportDir, segment_owner_to_dirname, ExcelReportRow, DAILY_NAME_FORMAT, daily_ttl, logger)
    monthly_ttl = datetime.timedelta(days=config.MonthlyReportTtlDays)
    monthly_report_generator = report_generator.ReportGenerator(config.MonthlyReportDir, segment_owner_to_dirname, ExcelReportRow, "%Y-%m", monthly_ttl, logger)

    tracker = ProcessedTablesTracker(config.FlattenedSharedRetargetingChevents)
    monthly_report_tracker = ProcessedTablesTracker(TTrackedSource(
        SourceDir=config.StatsDir,
        TrackTable=config.MonthlyTrackTable,
    ))

    while True:
        with yt_client.Transaction() as tx:
            for base_dir in (config.DailyReportDir, config.MonthlyReportDir):
                for dirname in segment_owner_to_dirname.values():
                    yt_client.mkdir(yt.ypath_join(base_dir, dirname), recursive=True)

            date = tables_helpers.get_unprocessed_date(yt_client, tracker, [config.FlattenedSharedMultipliersCheventsDir], logger)
            if date is None:
                break

            logger.info("Processing date %s", date)

            retargeting_table_path = ypath.ypath_join(config.FlattenedSharedRetargetingChevents.SourceDir, date)
            multipliers_table_path = ypath.ypath_join(config.FlattenedSharedMultipliersCheventsDir, date)

            stats_table_path = ypath.ypath_join(config.StatsDir, date)

            logger.info("Processing tables %s", [retargeting_table_path, multipliers_table_path])

            def tmp_table():
                return yt.TempTable(config.Yt.TmpDir, client=yt_client)

            with tmp_table() as dmp_chevents_table_path, tmp_table() as dmp_chevents_with_acl_table_path:
                fetch_dmp_segment_ids_from_audience_to_yt(yt_client, audience_client, segment_owners, config.DmpSegmentIdsTable, logger)
                get_dmp_chevents(config.Yt, tx, retargeting_table_path, multipliers_table_path, segment_owners, dmp_chevents_table_path, logger)
                join_acl.run(audience_client, config.Yt, yt_client, tx, dmp_chevents_table_path, dmp_chevents_with_acl_table_path, logger)
                calc_stats(config.Yt, tx, config.CryptaIdentifierUdfUrl, dmp_chevents_with_acl_table_path, config.DmpSegmentIdsTable, stats_table_path, logger)
                yt_helpers.set_ttl_by_table_name(stats_table_path, datetime.timedelta(days=config.StatsTtlDays), yt_client)

            daily_report_generator.generate_reports(yt_client, [stats_table_path], datetime.datetime.strptime(date, DAILY_NAME_FORMAT))

            tracker.add_processed_tables(yt_client, [retargeting_table_path])

    generate_monthly_reports(monthly_report_tracker, monthly_report_generator, yt_client, logger)


def fetch_dmp_segment_ids_from_audience_to_yt(yt_client, audience_client, segment_owners, destination_table_path, logger):
    with yt_client.Transaction():
        destination_table = yt.TablePath(destination_table_path, schema=schema_utils.get_schema_from_proto(TSegmentDmpId), attributes={"optimize_for": "scan"})
        yt_client.create("table", path=destination_table, recursive=True, force=True)
        rows = obtain_dmp_segment_ids(audience_client, segment_owners, logger)
        yt_client.write_table(destination_table, rows, format=JSON_FORMAT, raw=True)


def obtain_dmp_segment_ids(audience_client, segment_owners, logger):
    for audience_login in segment_owners:
        logger.info("Get segments of %s", audience_login)

        for segment in audience_client.list_segments(audience_login):
            if segment.get("source_name") == DMP_SEGMENT_SOURCE and segment["owner"] == audience_login:
                audience_segment_name = segment_name.deserialize(segment["name"])
                yield proto2json.proto2json(TSegmentDmpId(
                    SegmentId=segment["id"],
                    Title=audience_segment_name.title,
                    DmpSegmentId=audience_segment_name.aam_segment_id,
                    Owner=audience_login,
                ))


def get_dmp_chevents(yt_config, tx, retargeting_table_path, multipliers_table_path, segment_owners, output_table_path, logger):
    params = {
        "retargeting_table_path": retargeting_table_path,
        "multipliers_table_path": multipliers_table_path,
        "segment_owners": segment_owners,
        "output_table_path": output_table_path,
    }
    yql_helpers.run_query("/query/get_dmp_chevents.yql", yt_config, params, logger, tx=tx)


def calc_stats(yt_config, tx, crypta_identifier_udf_url, input_table_path, dmp_segment_ids_table_path, output_table_path, logger):
    params = {
        "crypta_identifier_udf_url": crypta_identifier_udf_url,
        "input_table_path": input_table_path,
        "dmp_segment_ids_table_path": dmp_segment_ids_table_path,
        "output_table_path": output_table_path,
        "stats": stats.get_stats_query(),
    }
    yql_helpers.run_query("/query/calc_dmp_per_segment_login_stats.yql", yt_config, params, logger, tx=tx)


def generate_monthly_reports(tracker, generator, yt_client, logger):
    logger.info("Generate monthly reports...")

    src_table_paths = tracker.get_unprocessed_tables(yt_client)
    full_months = month_grouper.get_full_months(src_table_paths)

    if not full_months:
        logger.info("No full months to process")
        return

    for item in full_months:
        with yt_client.Transaction():
            generator.generate_reports(yt_client, item.paths, datetime.datetime(year=item.year, month=item.month, day=1))
            tracker.add_processed_tables(yt_client, item.paths)


class ExcelReportRow(report_generator.ExcelReportRowBase):
    class Field(object):
        date = "Date"
        logins = "Login(s)"
        model = "Model"
        dmp_segment_id = "Segment ID"
        dmp_segment_name = "Segment Name"
        usage = "Usage"
        shows = "Shows"
        clicks = "Clicks"

    titles = [Field.date, Field.logins, Field.model, Field.dmp_segment_id, Field.dmp_segment_name, Field.usage, Field.shows, Field.clicks]
    column_width = {
        Field.date: 10,
        Field.logins: 40,
        Field.model: 10,
        Field.dmp_segment_id: 10,
        Field.dmp_segment_name: 80,
        Field.usage: 20,
        Field.shows: 10,
        Field.clicks: 10,
    }

    def __init__(self, item):
        self.date = item["date"]
        self.logins = item["logins_to_charge"]
        self.model = PAYMENT_MODEL[item["placeid"]]
        self.dmp_segment_id = item["dmp_segment_id"]
        self.dmp_segment_name = item["dmp_segment_name"].decode("utf-8")
        self.usage = item["usage"]
        self.shows = item["shows"]
        self.clicks = item["clicks"]

    @property
    def _mapping(self):
        return {
            self.Field.date: self.date,
            self.Field.logins: serialize_logins(self.logins),
            self.Field.model: self.model,
            self.Field.dmp_segment_id: self.dmp_segment_id,
            self.Field.dmp_segment_name: self.dmp_segment_name,
            self.Field.usage: self.usage,
            self.Field.shows: self.shows,
            self.Field.clicks: self.clicks,
        }

    def valid(self):
        return self.logins != []


def serialize_logins(logins):
    result = "; ".join(sorted(logins))
    return result if result else "N/A"
