import collections
import datetime

from crypta.buchhalter.services.main.lib.audience_per_segment_login_metrics.config_pb2 import TConfig
from crypta.buchhalter.services.main.lib.common.helpers import indexes
from crypta.lib.python.solomon import reporter
from crypta.lib.python.yt import (
    path_utils,
    yt_helpers,
)
from crypta.lib.python.yt.processed_tables_tracker import ProcessedTablesTracker


EMPTY_LOGINS_TO_CHARGE_METRIC = "empty_logins_to_charge"
TOTAL_METRIC = "total"


def add_args_subparser(subparsers):
    parser = subparsers.add_parser("audience_per_segment_login_metrics", help="Calc metrics for (s)dmp per segment-login stats")
    parser.set_defaults(function=run)
    parser.set_defaults(config_proto_cls=TConfig)


def run(config, logger):
    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)
    solomon_reporter = reporter.create_solomon_reporter_from_config(config.Solomon)

    dmp_tracker = ProcessedTablesTracker(config.DmpPerSegmentLoginStats)
    sdmp_tracker = ProcessedTablesTracker(config.SdmpPerSegmentLoginStats)

    dmp_segment_owners = indexes.get_audience_logins(yt_client, config.DmpIndexTable)
    sdmp_segment_owners = [item["AudienceLogin"] for item in yt_client.read_table(config.SdmpIndexTable)]

    process(dmp_tracker, dmp_segment_owners, yt_client, solomon_reporter, logger)
    process(sdmp_tracker, sdmp_segment_owners, yt_client, solomon_reporter, logger)


def process(tracker, segment_owners, yt_client, solomon_reporter, logger):
    while True:
        with yt_client.Transaction():
            src_table_paths = tracker.get_unprocessed_tables(yt_client, 1)

            if not src_table_paths:
                logger.info("No tables to process")
                break

            src_table_path = src_table_paths[0]

            logger.info("Processing table %s", src_table_path)

            dt = datetime.datetime.strptime(path_utils.get_basename(src_table_path), "%Y-%m-%d")

            result = _get_metrics(src_table_path, yt_client)

            for segment_owner in segment_owners:
                metrics = result[segment_owner]
                for metric, value in metrics.iteritems():
                    solomon_reporter.set_value(metric, value, labels={"segment_owner": segment_owner}, ts_datetime=dt)

            tracker.add_processed_tables(yt_client, [src_table_path])


def _default_metrics_factory():
    return {EMPTY_LOGINS_TO_CHARGE_METRIC: 0, TOTAL_METRIC: 0}


def _get_metrics(path, yt_client):
    result = collections.defaultdict(_default_metrics_factory)
    for row in yt_client.read_table(path):
        segment_owner = row["segment_owner"]
        result[segment_owner][TOTAL_METRIC] += 1
        result[segment_owner][EMPTY_LOGINS_TO_CHARGE_METRIC] += 1 if not row["logins_to_charge"] else 0
    return result
