import datetime

from dateutil import relativedelta
from library.python.protobuf.json import json2proto
import pytz
import yt.wrapper as yt
from yt.wrapper import ypath

from crypta.buchhalter.services.main.lib.audience_send_shadow_dmp_reports.config_pb2 import TConfig
from crypta.buchhalter.services.main.lib.common import report_mail_sender
from crypta.buchhalter.services.main.lib.common.proto.shadow_dmp_index_pb2 import TShadowDmpIndex
from crypta.lib.native.yt.processed_tables_tracker.proto.tracked_source_pb2 import TTrackedSource
from crypta.lib.python import time_utils
from crypta.lib.python.smtp.text_mail_sender import TextMailSender
from crypta.lib.python.yt import path_utils
from crypta.lib.python.yt import yt_helpers
from crypta.lib.python.yt.processed_tables_tracker import ProcessedTablesTracker


class DailySettings(object):
    date_format = "%Y-%m-%d"
    report_name_format = "%Y-%m-%d.xlsx"
    subject_template = "Yandex.Audience segments daily usage report {date}"


class MonthlySettings(object):
    date_format = "%Y-%m"
    report_name_format = "%Y-%m.xlsx"
    subject_template = "Yandex.Audience segments monthly usage report {date}"


def add_args_subparser(subparsers):
    parser = subparsers.add_parser("audience_send_shadow_dmp_reports", help="Send reports to shadow dmps")
    parser.set_defaults(function=run)
    parser.set_defaults(config_proto_cls=TConfig)


def run(config, logger):
    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)
    oldest_daily_report_date_to_send = time_utils.get_current_moscow_datetime() - datetime.timedelta(days=config.OldestDailyReportToSendDays)
    oldest_monthly_report_date_to_send = time_utils.get_current_moscow_datetime() - relativedelta.relativedelta(months=config.OldestMonthlyReportToSendMonths + 1)

    failed_shadow_dmps = []

    for node in [config.ProcessedDailyReportDir, config.ProcessedMonthlyReportDir]:
        yt_client.create("map_node", node, recursive=True, ignore_existing=True)

    for item in yt_client.read_table(config.ShadowDmpIndexTable, format=yt.JsonFormat(attributes={"encode_utf8": False}), raw=True):
        shadow_dmp = TShadowDmpIndex()
        json2proto.json2proto(item, shadow_dmp)

        logger.info("Process shadow dmp '%s'", shadow_dmp.AudienceLogin)
        daily_reports_tracker = ProcessedTablesTracker(TTrackedSource(
            SourceDir=ypath.ypath_join(config.DailyReportDir, shadow_dmp.Name),
            TrackTable=ypath.ypath_join(config.ProcessedDailyReportDir, shadow_dmp.Name),
        ))
        monthly_reports_tracker = ProcessedTablesTracker(TTrackedSource(
            SourceDir=ypath.ypath_join(config.MonthlyReportDir, shadow_dmp.Name),
            TrackTable=ypath.ypath_join(config.ProcessedMonthlyReportDir, shadow_dmp.Name),
        ))
        mail_sender = TextMailSender(
            host=config.Smtp.Host,
            port=config.Smtp.Port,
            default_from_addr=config.Smtp.EmailFrom,
            default_to_addrs=list(shadow_dmp.Emails),
            default_cc=list(config.Smtp.EmailCc),
            default_bcc=list(config.Smtp.EmailBcc),
            dry_run=config.Smtp.DryRun,
        )

        try:
            logger.info("Process daily reports...")
            process_shadow_dmp(daily_reports_tracker, yt_client, mail_sender, shadow_dmp.AudienceLogin, oldest_daily_report_date_to_send, DailySettings, logger)

            logger.info("Process monthly reports...")
            process_shadow_dmp(monthly_reports_tracker, yt_client, mail_sender, shadow_dmp.AudienceLogin, oldest_monthly_report_date_to_send, MonthlySettings, logger)
        except Exception:
            logger.exception("")
            failed_shadow_dmps.append(shadow_dmp.AudienceLogin)

    if failed_shadow_dmps:
        raise Exception("There're errors in processing")


def process_shadow_dmp(tracker, yt_client, mail_sender, audience_login, oldest_report_date_to_send, settings, logger):
    unprocessed_reports = tracker.get_unprocessed_tables(yt_client)

    if not unprocessed_reports:
        logger.info("No reports to process")
        return

    for unprocessed_report in unprocessed_reports:
        logger.info("Process file '%s'", unprocessed_report)
        with yt_client.Transaction():
            filename = path_utils.get_basename(unprocessed_report)
            report_date = datetime.datetime.strptime(filename, settings.report_name_format).replace(tzinfo=pytz.timezone("Europe/Moscow"))

            if report_date < oldest_report_date_to_send:
                logger.info("Too old report. Skip sending")
            else:
                logger.info("Download report and send")
                report_mail_sender.send(mail_sender, yt_client, unprocessed_report, report_date, audience_login, settings)

            tracker.add_processed_tables(yt_client, [unprocessed_report])
