import datetime

from library.python.protobuf.json import (
    json2proto,
    proto2json,
)
import yt.wrapper as yt
from yt.wrapper import ypath

from crypta.buchhalter.services.main.lib.audience_shadow_dmp_report_calc.config_pb2 import TConfig
from crypta.buchhalter.services.main.lib.common import (
    join_acl,
    report_generator,
    stats,
)
from crypta.buchhalter.services.main.lib.common.helpers import (
    indexes,
    month_grouper,
    tables_helpers,
)
from crypta.buchhalter.services.main.lib.common.proto.segment_name_pb2 import TSegmentName
from crypta.buchhalter.services.main.lib.common.proto.shadow_dmp_index_pb2 import TShadowDmpIndex
from crypta.lib.native.yt.processed_tables_tracker.proto.tracked_source_pb2 import TTrackedSource
import crypta.lib.python.audience.client as audience
from crypta.lib.python.identifiers import identifiers
from crypta.lib.python.yql import yql_helpers
from crypta.lib.python.yt import schema_utils
from crypta.lib.python.yt import yt_helpers
from crypta.lib.python.yt.processed_tables_tracker import ProcessedTablesTracker

DAILY_NAME_FORMAT = "%Y-%m-%d"
JSON_FORMAT = yt.JsonFormat(attributes={"encode_utf8": False})
PAYMENT_MODEL = {
    542: "CPC",
    1542: "CPM",
}


def add_args_subparser(subparsers):
    parser = subparsers.add_parser("audience_shadow_dmp_report_calc", help="Calc shadow dmp per segment-login report")
    parser.set_defaults(function=main)
    parser.set_defaults(config_proto_cls=TConfig)
    parser.add_argument("--index", required=True, type=indexes.get_shadow_dmp_index)


def main(config, logger, index):
    segment_owner_to_dirname = {item.AudienceLogin: item.Name for item in index}

    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)
    audience_client = audience.PublicApiAudienceClient.create_from_proto(config.Audience, logger)

    daily_ttl = datetime.timedelta(days=config.DailyReportTtlDays)
    daily_report_generator = report_generator.ReportGenerator(config.DailyReportDir, segment_owner_to_dirname, ExcelReportRow, DAILY_NAME_FORMAT, daily_ttl, logger)
    monthly_ttl = datetime.timedelta(days=config.MonthlyReportTtlDays)
    monthly_report_generator = report_generator.ReportGenerator(config.MonthlyReportDir, segment_owner_to_dirname, ExcelReportRow, "%Y-%m", monthly_ttl, logger)

    tracker = ProcessedTablesTracker(config.FlattenedSharedRetargetingChevents)
    monthly_report_tracker = ProcessedTablesTracker(TTrackedSource(
        SourceDir=config.ShadowDmpPerSegmentLoginStatsDir,
        TrackTable=config.MonthlyTrackTable,
    ))

    publish_shadow_dmp_index(yt_client, config.ShadowDmpIndexTable, index)

    while True:
        with yt_client.Transaction() as tx:
            date = tables_helpers.get_unprocessed_date(yt_client, tracker, [config.FlattenedSharedMultipliersCheventsDir], logger)
            if date is None:
                break

            logger.info("Processing date %s", date)

            retargeting_table_path = ypath.ypath_join(config.FlattenedSharedRetargetingChevents.SourceDir, date)
            multipliers_table_path = ypath.ypath_join(config.FlattenedSharedMultipliersCheventsDir, date)
            output_table_path = ypath.ypath_join(config.ShadowDmpPerSegmentLoginStatsDir, date)

            logger.info("Processing tables %s", [retargeting_table_path, multipliers_table_path])

            def tmp_table():
                return yt.TempTable(config.Yt.TmpDir, client=yt_client)

            with tmp_table() as sdmp_chevents_table_path, tmp_table() as sdmp_chevents_with_acl_table_path:
                get_segment_names(audience_client, yt_client, config.ShadowDmpIndexTable, config.ShadowDmpSegmentNamesTable, logger)
                get_sdmp_chevents(config.Yt, tx, retargeting_table_path, multipliers_table_path, config.ShadowDmpIndexTable, sdmp_chevents_table_path, logger)
                join_acl.run(audience_client, config.Yt, yt_client, tx, sdmp_chevents_table_path, sdmp_chevents_with_acl_table_path, logger)
                calc_stats(config.Yt, tx, config.CryptaIdentifierUdfUrl, sdmp_chevents_with_acl_table_path, config.ShadowDmpSegmentNamesTable, output_table_path, logger)
                yt_helpers.set_ttl_by_table_name(output_table_path, datetime.timedelta(days=config.StatsTtlDays), yt_client)

            daily_report_generator.generate_reports(yt_client, [output_table_path], datetime.datetime.strptime(date, DAILY_NAME_FORMAT))

            tracker.add_processed_tables(yt_client, [retargeting_table_path])

    generate_monthly_reports(monthly_report_tracker, monthly_report_generator, yt_client, logger)


def get_sdmp_chevents(yt_config, tx, retargeting_table_path, multipliers_table_path, index_table_path, output_table_path, logger):
    params = {
        "retargeting_table_path": retargeting_table_path,
        "multipliers_table_path": multipliers_table_path,
        "index_table_path": index_table_path,
        "output_table_path": output_table_path,
    }
    yql_helpers.run_query("/query/get_sdmp_chevents.yql", yt_config, params, logger, tx=tx)


def get_segment_names(audience_client, yt_client, index_table, output_table_path, logger):
    destination_table = yt.TablePath(output_table_path, schema=schema_utils.get_schema_from_proto(TSegmentName), attributes={"optimize_for": "scan"})
    yt_client.create("table", path=destination_table, recursive=True, ignore_existing=True)
    rows = obtain_segment_names(audience_client, yt_client, index_table, logger)
    yt_client.write_table(destination_table, rows, format=JSON_FORMAT, raw=True)


def obtain_segment_names(audience_client, yt_client, index_table, logger):
    for item in yt_client.read_table(index_table, format=JSON_FORMAT, raw=True):
        shadow_dmp = TShadowDmpIndex()
        json2proto.json2proto(item, shadow_dmp)
        logger.info("Get segments of %s", shadow_dmp.AudienceLogin)

        for segment in audience_client.list_segments(shadow_dmp.AudienceLogin):
            if identifiers.Login(segment["owner"]).normalize == shadow_dmp.AudienceLogin:
                yield proto2json.proto2json(TSegmentName(
                    SegmentId=segment["id"],
                    Title=segment["name"],
                    Owner=shadow_dmp.AudienceLogin,
                ))


def calc_stats(yt_config, tx, crypta_identifier_udf_url, input_table_path, segment_names_table_path, output_table_path, logger):
    params = {
        "crypta_identifier_udf_url": crypta_identifier_udf_url,
        "input_table_path": input_table_path,
        "segment_names_table_path": segment_names_table_path,
        "output_table_path": output_table_path,
        "stats": stats.get_stats_query(),
    }
    yql_helpers.run_query("/query/calc_shadow_dmp_per_segment_login_stats.yql", yt_config, params, logger, tx=tx)


def normalize_index_item(item):
    login = identifiers.Login(item.AudienceLogin)
    assert login.is_valid(), "Invalid index login '{}'".format(item.AudienceLogin)
    item.AudienceLogin = login.normalize
    return item


def publish_shadow_dmp_index(yt_client, index_table_path, index):
    schema = schema_utils.get_schema_from_proto(TShadowDmpIndex)

    with yt_client.Transaction():
        yt_client.create("table", index_table_path, recursive=True, force=True, attributes={"schema": schema})
        rows = (proto2json.proto2json(normalize_index_item(item)) for item in index)
        yt_client.write_table(index_table_path, rows, format=JSON_FORMAT, raw=True)


def generate_monthly_reports(tracker, generator, yt_client, logger):
    logger.info("Generate monthly reports...")

    src_table_paths = tracker.get_unprocessed_tables(yt_client)
    full_months = month_grouper.get_full_months(src_table_paths)

    if not full_months:
        logger.info("No full months to process")
        return

    for item in full_months:
        with yt_client.Transaction():
            generator.generate_reports(yt_client, item.paths, datetime.datetime(year=item.year, month=item.month, day=1))
            tracker.add_processed_tables(yt_client, item.paths)


class ExcelReportRow(report_generator.ExcelReportRowBase):
    class Field(object):
        date = "Date"
        logins = "Login(s)"
        model = "Model"
        segment_id = "Segment ID"
        segment_name = "Segment Name"
        usage = "Usage"
        shows = "Shows"
        clicks = "Clicks"

    titles = [Field.date, Field.logins, Field.model, Field.segment_id, Field.segment_name, Field.usage, Field.shows, Field.clicks]
    column_width = {
        Field.date: 10,
        Field.logins: 40,
        Field.model: 10,
        Field.segment_id: 10,
        Field.segment_name: 80,
        Field.usage: 20,
        Field.shows: 10,
        Field.clicks: 10,
    }

    def __init__(self, item):
        self.date = item["date"]
        self.logins = item["logins_to_charge"]
        self.model = PAYMENT_MODEL[item["placeid"]]
        self.segment_id = item["segmentid"]
        self.segment_name = item["segment_name"].decode("utf-8")
        self.usage = item["usage"]
        self.shows = item["shows"]
        self.clicks = item["clicks"]

    @property
    def _mapping(self):
        return {
            self.Field.date: self.date,
            self.Field.logins: serialize_logins(self.logins),
            self.Field.model: self.model,
            self.Field.segment_id: self.segment_id,
            self.Field.segment_name: self.segment_name,
            self.Field.usage: self.usage,
            self.Field.shows: self.shows,
            self.Field.clicks: self.clicks,
        }

    def valid(self):
        return self.logins != []


def serialize_logins(logins):
    result = "; ".join(sorted(logins))
    return result if result else "N/A"
