import datetime

import yaml
from yt.wrapper import ypath

from crypta.buchhalter.services.main.lib.common import stats
from crypta.buchhalter.services.main.lib.common.helpers import (
    indexes,
    tables_helpers,
)
from crypta.buchhalter.services.main.lib.audience_shadow_dmps_calc.config_pb2 import TConfig
from crypta.lib.python.yql import yql_helpers
from crypta.lib.python.yt import yt_helpers
from crypta.lib.python.yt.processed_tables_tracker import ProcessedTablesTracker


def add_args_subparser(subparsers):
    parser = subparsers.add_parser("audience_shadow_dmps_calc", help="Calculate shadow DMPs stats")
    parser.set_defaults(function=calc_shadow_dmps_stats)
    parser.set_defaults(config_proto_cls=TConfig)
    parser.add_argument("--dmp-index", required=True)
    parser.add_argument("--adobe-index", required=True)
    parser.add_argument("--block-lists", required=True)


def calc_shadow_dmps_stats(config, logger, dmp_index, adobe_index, block_lists):
    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)

    tracker = ProcessedTablesTracker(config.TargetingFlattenedSharedChevents)

    explicit_block_list, expr_block_list = get_block_lists(block_lists, dmp_index, adobe_index)
    logger.info("Explicit logins block list: %s\n Expressions block list: %s", explicit_block_list, expr_block_list)

    while True:
        with yt_client.Transaction() as tx:
            date = tables_helpers.get_unprocessed_date(yt_client, tracker, [config.MultipliersFlattenedSharedCheventsDir], logger)
            if date is None:
                break

            targeting_flattened_shared_chevents_table_path = ypath.ypath_join(config.TargetingFlattenedSharedChevents.SourceDir, date)
            multipliers_flattened_shared_chevents_table_path = ypath.ypath_join(config.MultipliersFlattenedSharedCheventsDir, date)

            logger.info("Processing tables %s and %s", targeting_flattened_shared_chevents_table_path, multipliers_flattened_shared_chevents_table_path)

            targeting_stats_table_path = ypath.ypath_join(config.TargetingStatsDir, date)
            multipliers_stats_table_path = ypath.ypath_join(config.MultipliersStatsDir, date)
            total_stats_table_path = ypath.ypath_join(config.TotalStatsDir, date)

            run_query(
                config.Yt,
                tx,
                targeting_flattened_shared_chevents_table_path,
                multipliers_flattened_shared_chevents_table_path,
                targeting_stats_table_path=targeting_stats_table_path,
                multipliers_stats_table_path=multipliers_stats_table_path,
                total_stats_table_path=total_stats_table_path,
                explicit_block_list=explicit_block_list,
                expr_block_list=expr_block_list,
                logger=logger,
            )

            ttl_timedelta = datetime.timedelta(days=config.OutputTtlDays)
            for table_path in (targeting_stats_table_path, multipliers_stats_table_path, total_stats_table_path):
                yt_helpers.set_ttl_by_table_name(table_path, ttl_timedelta, yt_client)

            tracker.add_processed_tables(yt_client, [targeting_flattened_shared_chevents_table_path])


def run_query(
    yt_config,
    tx,
    targeting_flattened_shared_chevents_table_path,
    multipliers_flattened_shared_chevents_table_path,
    targeting_stats_table_path,
    multipliers_stats_table_path,
    total_stats_table_path,
    explicit_block_list,
    expr_block_list,
    logger,
):

    params = {
        "targeting_flattened_shared_chevents_table_path": targeting_flattened_shared_chevents_table_path,
        "multipliers_flattened_shared_chevents_table_path": multipliers_flattened_shared_chevents_table_path,

        "targeting_stats_table_path": targeting_stats_table_path,
        "multipliers_stats_table_path": multipliers_stats_table_path,
        "total_stats_table_path": total_stats_table_path,

        "stats": stats.get_stats_query(),
        "serialize_selecttype_func": stats.SERIALIZE_SELECTTYPE,
        "serialize_placeid_func": stats.SERIALIZE_PLACEID,
        "explicit_block_list": explicit_block_list,
        "expr_block_list": expr_block_list,
    }

    yql_helpers.run_query("/query/calc_shadow_dmps_stats.yql", yt_config, params, logger, tx=tx)


def get_block_lists(block_lists_filename, dmp_index_filename, adobe_index_filename):
    with open(block_lists_filename) as block_lists_file:
        content = yaml.safe_load(block_lists_file)

        explicit_block_list = content["logins"]
        explicit_block_list.extend(indexes.get_dmp_logins(dmp_index_filename))
        explicit_block_list.extend(indexes.get_adobe_logins(adobe_index_filename))

        return explicit_block_list, content["expressions"]
