import collections
import json
import os

from library.python import resource
import xlsxwriter
from yt.wrapper import ypath

from crypta.lib.python.yt import yt_helpers


EMPTY_REPORT_ATTRIBUTE = "empty_report"


class ReportGenerator(object):
    def __init__(self, output_dir, segment_owner_to_dirname, excel_report_row_class, name_format, ttl_timedelta, logger):
        self.output_dir = output_dir
        self.segment_owner_to_dirname = segment_owner_to_dirname
        self.excel_report_row_class = excel_report_row_class
        self.name_format = name_format
        self.ttl_timedelta = ttl_timedelta
        self.logger = logger

    def generate_reports(self, yt_client, input_table_paths, report_date):
        self.logger.info("Process table(s) %s", input_table_paths)

        filename_without_extention = report_date.strftime(self.name_format)
        filename = "{}.xlsx".format(filename_without_extention)

        grouped_rows = _group_by_segment_owner(input_table_paths, yt_client)
        for segment_owner, segment_owner_dirname in self.segment_owner_to_dirname.iteritems():
            self.logger.info("Generate report for %s", segment_owner)

            segment_owner_dir = ypath.ypath_join(self.output_dir, segment_owner_dirname)
            yt_client.create("map_node", segment_owner_dir, recursive=True, ignore_existing=True)

            data = grouped_rows[segment_owner]
            self.create_excel_report(filename, data)

            yt_file_path = ypath.ypath_join(segment_owner_dir, filename)
            self.logger.info("Upload report to %s", yt_file_path)

            with open(filename) as f:
                yt_client.write_file(yt_file_path, f)

            yt_helpers.set_attribute(yt_file_path, EMPTY_REPORT_ATTRIBUTE, data == [], client=yt_client)
            yt_helpers.set_ttl_by_table_name(yt_file_path, self.ttl_timedelta, yt_client, name_format="{}.xlsx".format(self.name_format))

            os.remove(filename)

    def create_excel_report(self, filename, data):
        excel_report_rows = sorted([self.excel_report_row_class(item) for item in data], key=lambda x: x.to_tuple())
        excel_report_rows = [x for x in excel_report_rows if x.valid()]

        with xlsxwriter.Workbook(filename) as workbook:
            title_format = workbook.add_format(json.loads(resource.find("/styles/title_style.json")))
            cell_format = workbook.add_format(json.loads(resource.find("/styles/cell_style.json")))

            worksheet = workbook.add_worksheet("Report")

            for column, field in enumerate(self.excel_report_row_class.titles):
                worksheet.set_column(column, column, self.excel_report_row_class.column_width[field])

            worksheet.write_row(0, 0, self.excel_report_row_class.titles, title_format)

            for row_number, row in enumerate(excel_report_rows):
                worksheet.write_row(row_number + 1, 0, row.to_tuple(), cell_format)


class ExcelReportRowBase(object):
    class Field(object):
        pass

    titles = []
    column_width = {}

    def __init__(self, item):
        raise NotImplementedError

    @property
    def _mapping(self):
        return {}

    def to_tuple(self):
        return tuple(self._mapping[field] for field in self.titles)

    def valid(self):
        return True


def _group_by_segment_owner(paths, yt_client):
    result = collections.defaultdict(list)

    for path in paths:
        for row in yt_client.read_table(path):
            segment_owner = row["segment_owner"]
            result[segment_owner].append(row)

    return result


def is_report_empty(path, yt_client):
    return yt_helpers.has_attribute(path, EMPTY_REPORT_ATTRIBUTE, yt_client) and yt_helpers.get_attribute(path, EMPTY_REPORT_ATTRIBUTE, yt_client)
