import datetime

from yt.wrapper import ypath

from crypta.buchhalter.services.main.lib.common import stats
from crypta.buchhalter.services.main.lib.common.helpers import tables_helpers
from crypta.buchhalter.services.main.lib.direct_calc.config_pb2 import TConfig
from crypta.lib.python.yql import yql_helpers
from crypta.lib.python.yt import path_utils
from crypta.lib.python.yt import yt_helpers
from crypta.lib.python.yt.processed_tables_tracker import ProcessedTablesTracker


def add_args_subparser(subparsers):
    parser = subparsers.add_parser("direct_calc", help="Calc retargeting stats in Yandex.Direct")
    parser.set_defaults(function=calc_stats)
    parser.set_defaults(config_proto_cls=TConfig)


def calc_stats(config, logger):
    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)

    tracker = ProcessedTablesTracker(config.CheventLog)

    while True:
        with yt_client.Transaction() as tx:
            date = tables_helpers.get_unprocessed_date(yt_client, tracker, [config.ActionCheckedLogDir, config.GoalContextDir], logger)
            if date is None:
                break

            logger.info("Processing date %s", date)

            chevent_table_path = ypath.ypath_join(config.CheventLog.SourceDir, date)
            retargeting_chevents_table_path = ypath.ypath_join(config.RetargetingCheventsDir, date)
            retargeting_stats_table_path = ypath.ypath_join(config.RetargetingStatsDir, date)

            params = {
                "chevent_table_path": chevent_table_path,
                "action_checked_log_table_path": ypath.ypath_join(config.ActionCheckedLogDir, date),
                "goal_context_table_path": ypath.ypath_join(config.GoalContextDir, date),
                "retargeting_chevents_table_path": retargeting_chevents_table_path,
                "retargeting_stats_table_path": retargeting_stats_table_path,
                "stats": stats.get_stats_query(),
            }

            yql_helpers.run_query("/query/calc_direct_stats.yql", config.Yt, params, logger, tx=tx)

            ttl_timedelta = datetime.timedelta(days=config.OutputTtlDays)
            for table_path in (retargeting_chevents_table_path, retargeting_stats_table_path):
                yt_helpers.set_ttl_by_table_name(table_path, ttl_timedelta, yt_client)

            logger.info("RetargetingCheventsDir content: %s", yt_client.list(config.RetargetingCheventsDir))
            tracker.add_processed_tables(yt_client, [chevent_table_path])


def get_date(unprocessed_table_paths, action_checked_log_tables):
    for unprocessed_table_path in unprocessed_table_paths:
        date = path_utils.get_basename(unprocessed_table_path)

        if date in action_checked_log_tables:
            return date

    return None
