import datetime

from library.python import resource
import yaml

from crypta.dmp.common import index_status
from crypta.lib.python import templater
from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler import (
    consts,
    juggler_check_generator,
)
from crypta.lib.python.spine.sandbox import sandbox_scheduler
from sandbox.projects.crypta import buchhalter


HOUR_1 = datetime.timedelta(hours=1)
HOURS_3 = datetime.timedelta(hours=3)
HOURS_12 = datetime.timedelta(hours=12)
HOURS_24 = datetime.timedelta(hours=24)
HOURS_36 = datetime.timedelta(hours=36)
HOURS_48 = datetime.timedelta(hours=48)


class Scheduler:
    def __init__(self, task_name, crit_time, kill_timeout, schedule_interval=HOUR_1):
        self.task_name = task_name
        self.crit_time = crit_time
        self.kill_timeout = kill_timeout
        self.schedule_interval = schedule_interval


def get_registry():
    juggler_host = "crypta-buchhalter"
    juggler = juggler_check_generator.CryptaYtCheckGenerator(tags=["crypta-buchhalter"], host=juggler_host, child_group_type=consts.GroupType.host, child_group="{}-raw".format(juggler_host))

    sandbox = sandbox_scheduler.create_default_generator(juggler, ["BUCHHALTER"])

    for item in [
        Scheduler(buchhalter.TaskName.adfox_calc, crit_time=HOURS_36, kill_timeout=HOURS_48),
        Scheduler(buchhalter.TaskName.backuper, crit_time=HOURS_36, kill_timeout=HOURS_3, schedule_interval=HOURS_12),
        Scheduler(buchhalter.TaskName.direct_calc, crit_time=HOURS_36, kill_timeout=HOURS_48),
        Scheduler(buchhalter.TaskName.display_calc, crit_time=HOURS_36, kill_timeout=HOURS_48),
        Scheduler(buchhalter.TaskName.lookalike_calc, crit_time=HOURS_36, kill_timeout=HOURS_24),
        Scheduler(buchhalter.TaskName.multipliers_calc, crit_time=HOURS_36, kill_timeout=HOURS_24),
        Scheduler(buchhalter.TaskName.audience_per_owner_calc, crit_time=HOURS_36, kill_timeout=HOURS_24),
        Scheduler(buchhalter.TaskName.audience_add_logins_to_segments, crit_time=HOURS_12, kill_timeout=HOURS_3),
        Scheduler(buchhalter.TaskName.audience_group_segments, crit_time=HOURS_12, kill_timeout=HOURS_3),
        Scheduler(buchhalter.TaskName.audience_per_segment_events_calc, crit_time=HOURS_36, kill_timeout=HOURS_24),
        Scheduler(buchhalter.TaskName.audience_per_segment_login_metrics, crit_time=HOURS_48, kill_timeout=HOUR_1, schedule_interval=HOURS_24),
        Scheduler(buchhalter.TaskName.audience_per_group_calc, crit_time=HOURS_36, kill_timeout=HOURS_24),
        Scheduler(buchhalter.TaskName.audience_filter_shared_chevents, crit_time=HOURS_36, kill_timeout=HOURS_24),
        Scheduler(buchhalter.TaskName.audience_shadow_dmps_calc, crit_time=HOURS_36, kill_timeout=HOURS_24),
        Scheduler(buchhalter.TaskName.audience_dmp_report_calc, crit_time=HOURS_36, kill_timeout=HOURS_3),
        Scheduler(buchhalter.TaskName.audience_shadow_dmp_report_calc, crit_time=HOURS_36, kill_timeout=HOURS_3),
        Scheduler(buchhalter.TaskName.audience_send_shadow_dmp_reports, crit_time=HOURS_48, kill_timeout=HOURS_3),
    ]:
        task_cls = buchhalter.CryptaBuchhalterTask
        sandbox.create_scheduler(
            task_cls=task_cls,
            extra_params={
                "task_name": item.task_name,
            },
            schedule_interval=item.schedule_interval,
            kill_timeout=item.kill_timeout,
        ).check(
            crit_time=item.crit_time,
            juggler_service=task_cls.get_class_juggler_service(item.task_name),
        ).add_yt_dependencies(
            YtProxy.Group.offline,
        )

    return juggler


def _get_dmp_index():
    index = yaml.safe_load(templater.render_resource("/buchhalter/dmp_index", vars={"environment": environment.STABLE}, strict=True))
    return [dmp for dmp in index if dmp["status"] == index_status.ACTIVE and not dmp.get("internal")]


def _get_sdmp_index():
    return yaml.safe_load(resource.find("/buchhalter/sdmp_index"))
