#include "parse_cm_access_log_error_fields.h"
#include "parse_cm_access_log_mapper.h"

#include <crypta/cm/services/api/lib/logic/upload/request/upload_request_parser.h>
#include <crypta/cm/services/common/data/id_utils.h>
#include <crypta/cm/offline/bin/common/match_fields.h>
#include <crypta/lib/native/access_log/access_log_fields.h>

#include <mapreduce/yt/interface/node.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/http/io/stream.h>
#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/yson/node/node_io.h>

#include <util/stream/str.h>

using namespace NCrypta;
using namespace NCrypta::NOfflineCm;
using namespace NCrypta::NOfflineCm::NMatchFields;
using namespace NCrypta::NOfflineCm::NParseCmAccessLogErrorFields;

namespace {
    bool IsSuccessfulUpload(const TStringBuf& requestPath, ui64 httpCode) {
        return NCm::NApi::NUploadRequestParser::IsUploadRequestPath(requestPath) && httpCode == 200;
    }

    ui64 GetHttpCode(const NYT::TNode& row) {
        const auto& map = row.AsMap();
        const auto& it = map.find(NAccessLogFields::REPLY);
        if (it != map.end() && !it->second.IsNull()) {
            return ParseHttpRetCode(it->second.AsString());
        }

        return map.at(NAccessLogFields::HTTP_CODE).AsUint64();
    }
}

TParseCmAccessLogMapper::TParseCmAccessLogMapper(const TOutputIndexes& outputIndexes)
    : OutputIndexes(outputIndexes)
{
}

void TParseCmAccessLogMapper::Do(TReader* reader, TWriter* writer) {
    for (; reader->IsValid(); reader->Next()) {
        const auto& row = reader->GetRow();

        try {
            TParsedHttpLocation httpLocation(row.At(NAccessLogFields::QUERY).AsString());

            if (!IsSuccessfulUpload(httpLocation.Path, GetHttpCode(row))) {
                continue;
            }

            const auto& unixtime = row.At(NAccessLogFields::UNIXTIME).AsUint64();
            const auto& body = Base64Decode(row.At(NAccessLogFields::BODY).AsString());
            const auto& request = NCm::NApi::NUploadRequestParser::Parse(TCgiParameters(httpLocation.Cgi), body);
            const auto& extId = request.Match.GetExtId();
            const auto* matchedId = request.Match.FindInternalId(NCm::YANDEXUID_TYPE);

            if (matchedId != nullptr) {
                const auto result = NYT::TNode()
                    (TIMESTAMP, unixtime)
                    (TAG, extId.Type)
                    (EXT_ID, extId.Value)
                    (YANDEXUID, matchedId->GetId().Value);
                writer->AddRow(result, *OutputIndexes[EOutputTables::Matches]);
            }
        } catch (const yexception& e) {
            const auto error = NYT::TNode()
                (ROW, row)
                (WHAT, e.what());
            writer->AddRow(error, *OutputIndexes[EOutputTables::Errors]);
        }
    }
}

REGISTER_MAPPER(TParseCmAccessLogMapper);
