#include "update_state_job.h"
#include "update_state_reducer.h"

#include <crypta/lib/native/range/packs.h>
#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/cm/offline/bin/common/match_fields.h>
#include <crypta/cm/offline/bin/common/match_schema.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/temp_table.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <util/string/join.h>

using namespace NCrypta::NOfflineCm;
using namespace NMatchFields;

TUpdateStateJob::TUpdateStateJob(const TConfig& config, NLog::TLogPtr log)
    : Config(config)
    , Log(log)
{
    Y_ENSURE(log != nullptr);
}

int TUpdateStateJob::Do() {
    Log->info("================ Start ================");

    auto client = NYT::CreateClient(Config.YtProxy);
    NYT::TConfig::Get()->Pool = Config.YtPool;

    Log->info("Look for source tables in {}", Config.SourceDir);
    const auto& sourceTables = List(client, Config.SourceDir, TListOptions().Absolute(true));
    Log->info("List of source tables [\n{}\n]", JoinSeq(",\n", sourceTables));

    for (const auto& pack: GetPacks(sourceTables, Config.TablePackSize)) {
        auto tx = client->StartTransaction();

        Log->info("Create state table {} if it not exists", Config.DestinationTable);
        tx->Create(Config.DestinationTable, NYT::NT_TABLE, NYT::TCreateOptions().Recursive(true).IgnoreExisting(true).Attributes(NYT::TNode()("schema", GetMatchSchemaWithUniqueKeys().ToNode())));

        Log->info("Update state table {} with fresh tables [\n{}\n]", Config.DestinationTable, JoinSeq(",\n", pack));

        {
            NYT::TTempTable tmp(tx);

            auto sortSpec = NYT::TSortOperationSpec()
                .Output(NYT::TRichYPath(tmp.Name()).Schema(GetMatchSchema()))
                .SortBy({EXT_ID, TIMESTAMP});
            AddInputs(sortSpec, pack);
            tx->Sort(sortSpec);

            auto reduceSpec = NYT::TReduceOperationSpec()
                .AddInput<NYT::TNode>(tmp.Name())
                .AddInput<NYT::TNode>(Config.DestinationTable)
                .AddOutput<NYT::TNode>(NYT::TRichYPath(Config.DestinationTable).Schema(GetMatchSchemaWithUniqueKeys()).OptimizeFor(NYT::OF_SCAN_ATTR))
                .ReduceBy(EXT_ID)
                .SortBy({EXT_ID, TIMESTAMP});
            tx->Reduce(reduceSpec, new TUpdateStateReducer(Config.Ttl));
        }

        if (Config.ShouldDropInput) {
            Log->info("Remove fresh tables [\n{}\n]", JoinSeq(",\n", pack));
            RemoveTables(tx, pack);
        }

        tx->Commit();
    }

    Log->info("================ Finish ===============");
    return 0;
}
