import datetime
import os

import yaml

from crypta.lib.python import templater
from crypta.lib.python.data_size import DataSize
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler import juggler_check_generator
from crypta.lib.python.spine.sandbox import sandbox_scheduler
from crypta.lib.python.spine.yt import yt_config_registry
from crypta.lib.python.spine.yt.yt_latency_metric import YtLatencyMetric
from crypta.lib.python.spine.yt.yt_processed_tables_metric import YtProcessedTablesMetric
from crypta.lib.python.spine.yt.yt_size_metric import YtSizeMetric
from sandbox.projects.crypta.cm.offline import (
    parse_cm_access_log,
    split_by_tag,
    update_state,
)


def render_path(tag, suffix):
    return os.path.join("cookie_matching/tags", tag, suffix)


def get_registry():
    stable_index = yaml.safe_load(templater.render_resource("/cm/offline_index", vars={"environment": "stable"}, strict=True))
    testing_index = yaml.safe_load(templater.render_resource("/cm/offline_index", vars={"environment": "testing"}, strict=True))

    juggler = juggler_check_generator.CryptaYtCheckGenerator(tags=["crypta-offline-cm"], host="crypta-offline-cm")
    sandbox = sandbox_scheduler.create_default_generator(juggler.clone(yt_dependencies=YtProxy.Group.offline), ["CM"])
    yt_registry = yt_config_registry.CryptaYtConfigRegistry(juggler)

    YtSizeMetric(yt_registry, "cookie_matching/fresh")
    YtSizeMetric(yt_registry, "cookie_matching/errors/parse_cm_access_log").add_disk_space_alert(
        predicate=alert_pb2.GT,
        threshold=DataSize(b=0),
    )

    YtLatencyMetric(yt_registry, "cookie_matching/fresh").add_latency_alert(
        threshold=datetime.timedelta(hours=4),
    )

    YtProcessedTablesMetric(yt_registry, "cookie_matching/processed_cm_access_log").add_lag_alert(
        datetime.timedelta(hours=4),
    )

    for item in stable_index:
        tag = item["tag"]
        fresh_dir = render_path(tag, "fresh")

        YtSizeMetric(yt_registry, render_path(tag, "state"))

        YtSizeMetric(yt_registry, fresh_dir).add_disk_space_alert(
            predicate=alert_pb2.LTE,
            threshold=DataSize(b=0),
            period=datetime.timedelta(days=7),
        )

        YtLatencyMetric(yt_registry, fresh_dir).add_latency_alert(
            threshold=datetime.timedelta(hours=4),
        )

        sandbox.create_scheduler(
            update_state.CryptaOfflineCmUpdateStateTask,
            schedule_interval=datetime.timedelta(hours=1),
            kill_timeout=datetime.timedelta(hours=2),
            env=environment.STABLE,
            extra_params=item,
        ).check(
            datetime.timedelta(hours=16),
            juggler_service=update_state.CryptaOfflineCmUpdateStateTask.get_juggler_service_for_tag(tag),
        )

    for item in testing_index:
        sandbox.create_scheduler(
            update_state.CryptaOfflineCmUpdateStateTask,
            schedule_interval=datetime.timedelta(hours=4),
            kill_timeout=datetime.timedelta(hours=2),
            env=environment.TESTING,
            extra_params=item,
        )

    for task in (
        parse_cm_access_log.CryptaOfflineCmParseCmAccessLogTask,
        split_by_tag.CryptaOfflineCmSplitByTagTask,
    ):
        sandbox.create_scheduler(
            task,
            schedule_interval=datetime.timedelta(minutes=30),
            kill_timeout=datetime.timedelta(hours=3),
            env=environment.STABLE,
        ).check(datetime.timedelta(hours=16))

        sandbox.create_scheduler(
            task,
            schedule_interval=datetime.timedelta(minutes=30),
            kill_timeout=datetime.timedelta(hours=3),
            env=environment.TESTING,
        )

    return juggler
