#include "mutation_sender.h"

#include <crypta/cm/services/common/changes/delete_command.h>
#include <crypta/cm/services/common/changes/expire_command.h>
#include <crypta/cm/services/common/changes/touch_command.h>
#include <crypta/cm/services/common/changes/upload_command.h>
#include <crypta/cm/services/common/data/id_hash.h>
#include <crypta/cm/services/common/data/id_utils.h>

using namespace NCrypta::NCm;
using namespace NCrypta::NCm::NApi;

namespace {
    const THash<TString> HASH;

    TString GetShardingKey(const TMatch& match) {
        Y_ENSURE(!match.GetInternalIds().empty(), "Match has no internal ids");

        const auto* yuid = match.FindInternalId(YANDEXUID_TYPE);
        if (yuid) {
            return yuid->GetId().Value;
        }
        const auto* icookie = match.FindInternalId(ICOOKIE_TYPE);
        if (icookie) {
            return icookie->GetId().Value;
        }

        return match.GetInternalIds().begin()->second.GetId().Value;
    }
}

TMutationSender::TMutationSender(NCrypta::NPQ::TProducer& producer)
    : Producer(producer)
{
}

void TMutationSender::UpdateMatch(const TMatch& match) {
    const auto& shardingKey = GetShardingKey(match);
    Producer.TryEnqueue(HASH(shardingKey), TUploadCommand::ToString(TUploadCommand(shardingKey, match)));
}

void TMutationSender::TouchMatch(const TMatch& match, TInstant touchTime) {
    const auto& shardingKey = GetShardingKey(match);
    Producer.TryEnqueue(HASH(shardingKey), TTouchCommand::ToString(TTouchCommand(shardingKey, match.GetExtId(), touchTime)));
}

void TMutationSender::DeleteMatch(const TMatch& match) {
    const auto& shardingKey = GetShardingKey(match);
    Producer.TryEnqueue(HASH(shardingKey), TDeleteCommand(shardingKey, match.GetExtId()).ToString());
}

void TMutationSender::ExpireMatch(const TMatch& match) {
    const auto& shardingKey = GetShardingKey(match);
    Producer.TryEnqueue(HASH(shardingKey), TExpireCommand(shardingKey, match.GetExtId()).ToString());
}
