#include "identify_processor.h"

#include <crypta/cm/services/api/lib/logic/identify/request/identify_request_parser.h>
#include <crypta/cm/services/common/data/id_utils.h>
#include <crypta/cm/services/common/db_state/get_changes_batch.h>
#include <crypta/cm/services/common/serializers/id/string/id_string_serializer.h>
#include <crypta/cm/services/common/serializers/match/record/match_record_serializer.h>
#include <crypta/cm/services/common/serializers/reply/json/reply_json_serializer.h>
#include <crypta/lib/native/singleton/tagged_singleton.h>
#include <crypta/lib/native/time/scope_timer.h>

#include <util/string/builder.h>

using namespace NCrypta::NCm;
using namespace NCrypta::NCm::NApi;

TIdentifyProcessor::TIdentifyProcessor(
        NYtDynTables::TKvDatabase& replicaDatabase,
        const TStats::TSettings& statsSettings,
        const TTurboDecryptor& turboDecryptor)
    : TRequestProcessor(NCrypta::NLog::GetLog("identify"), TaggedSingleton<TStats, decltype(*this)>("identify", statsSettings))
    , ReplicaDatabase(replicaDatabase)
    , TurboDecryptor(turboDecryptor)
{
}

void TIdentifyProcessor::DoProcess(NHttp::TRequestReply& reply, const TClient& clientInfo) {
    TScopeTimer scopeTimer(Stats.Percentile, "timing.process");

    Stats.Count->Add("request.total.received");
    Stats.Count->Add("tvm_client." + clientInfo.GetName());

    TIdentifyRequest request;
    try {
        request = NIdentifyRequestParser::Parse(TCgiParameters(reply.GetRequestCgi()));
    } catch (const yexception& e) {
        SendResponse(reply, HTTP_BAD_REQUEST, e.what());
        return;
    }

    Stats.Count->Add("client." + clientInfo.GetName() + "." + request.Subclient);

    const auto& requestedId = request.Id;

    if (IsTurboId(requestedId)) {
        RespondOnTurboId(reply, requestedId);
        return;
    }

    const auto& isInternalId = IsInternalId(requestedId);

    const THashSet<TString> trackedBackRefTags;
    const auto dbState = TDbStateLoader(Log, trackedBackRefTags).Load(ReplicaDatabase, {requestedId}, false, isInternalId);

    if (isInternalId) {
        RespondOnInternalId(reply, dbState, requestedId, clientInfo.GetPermissions().GetIdentifyRealtimeForbidden());
    } else {
        RespondOnExternalId(reply, dbState, requestedId, clientInfo.GetPermissions().GetIdentifyRealtimeForbidden());
    }
}

void TIdentifyProcessor::RespondOnNotFound(NHttp::TRequestReply& reply, const TDbState& dbState, const TId& requestedId) {
    Stats.Count->Add("request.status.not_found");
    if (dbState.GetBackRefs().Get(requestedId)) {
        const TString error = ::TStringBuilder() << "Missing match record for id " << NIdSerializer::ToString(requestedId);
        SendResponse(reply, HTTP_NOT_FOUND, error);
        Stats.Count->Add("errors.missing_match");
        Log->error(error);
    } else {
        SendResponse(reply, HTTP_NOT_FOUND, ::TStringBuilder() << "Nothing found for id " << NIdSerializer::ToString(requestedId));
    }
}

void TIdentifyProcessor::RespondOnExternalId(NHttp::TRequestReply& reply, const TDbState& dbState, const TId& requestedId, bool realtimeForbidden) {
    const auto* match = dbState.GetMatches().Get(requestedId);
    if (!match) {
        RespondOnNotFound(reply, dbState, requestedId);
        return;
    }

    const auto& internalIds = realtimeForbidden ? GetOfflineInternalIds(*match) : match->GetInternalIds();
    if (internalIds.empty()) {
        RespondOnNotFound(reply, dbState, requestedId);
        return;
    }

    SendResponse(reply, HTTP_OK, NReplyJsonSerializer::Serialize(internalIds));
}

void TIdentifyProcessor::RespondOnInternalId(NHttp::TRequestReply& reply, const TDbState& dbState, const TId& requestedId, bool realtimeForbidden) {
    const auto* backRef = dbState.GetBackRefs().Get(requestedId);
    if (!backRef) {
        RespondOnNotFound(reply, dbState, requestedId);
        return;
    }

    TMatchedIds response;
    for (const auto& id: backRef->Refs) {
        const auto* match = dbState.GetMatches().Get(id);
        if (!match) {
            continue;
        }

        auto& matchedIds = match->GetInternalIds();

        auto it = FindIf(matchedIds, [&requestedId](const auto& pair) { return pair.second.GetId() == requestedId; });

        if (it == matchedIds.end()) {
            Stats.Count->Add("errors.missing_id_in_record");
            Log->error("Missing id {} in {}", NIdSerializer::ToString(requestedId), NIdSerializer::ToString(id));
            continue;
        }

        auto responseMatchedId = it->second;

        if (realtimeForbidden && responseMatchedId.IsRealtime()) {
            Stats.Count->Add("request.realtime_forbidden.internal");
            continue;
        }

        responseMatchedId.SetId(id);
        response.emplace_back(std::move(responseMatchedId));
    }

    if (response.empty()) {
        RespondOnNotFound(reply, dbState, requestedId);
    } else {
        SendResponse(reply, HTTP_OK, NReplyJsonSerializer::Serialize(response));
    }
}

void TIdentifyProcessor::RespondOnTurboId(NHttp::TRequestReply& reply, const TId& requestedId) {
    const auto& decrypted = TurboDecryptor.Decrypt(requestedId.Value);

    if (!decrypted) {
        Stats.Count->Add("turbo.error");
        SendResponse(reply, HTTP_BAD_REQUEST, "Failed to decrypt cookie");
        return;
    }

    Stats.Count->Add("turbo.decrypted");
    SendResponse(reply, HTTP_OK, NReplyJsonSerializer::Serialize({TMatchedId(TId(YANDEXUID_TYPE, decrypted->Yandexuid), TInstant::Zero(), 0, {{"domain", decrypted->Domain}})}));
}

TMatch::TMatchedIds TIdentifyProcessor::GetOfflineInternalIds(const TMatch& match) {
    TMatch::TMatchedIds result;
    for (const auto& [idType, matchedId]: match.GetInternalIds()) {
        if (matchedId.IsRealtime()) {
            Stats.Count->Add("request.realtime_forbidden.external");
        } else {
            result[idType] = matchedId;
        }
    }
    return result;
}