#include "ping_processor.h"

#include <crypta/cm/services/api/lib/logic/ping/request/ping_request_parser.h>
#include <crypta/cm/services/common/data/id.h>
#include <crypta/cm/services/common/ping/constants.h>
#include <crypta/cm/services/common/serializers/id/string/id_string_serializer.h>
#include <crypta/cm/services/common/serializers/match/json/match_json_serializer.h>
#include <crypta/cm/services/common/serializers/reply/json/reply_json_serializer.h>
#include <crypta/lib/native/singleton/tagged_singleton.h>
#include <crypta/lib/native/time/scope_timer.h>

#include <util/string/builder.h>

using namespace NCrypta::NCm;
using namespace NCrypta::NCm::NApi;

TPingProcessor::TPingProcessor(
        NYtDynTables::TKvDatabase& replicaDatabase,
        const TPingConfig& pingConfig,
        const NTvmAuth::TTvmClient& tvmClient,
        const TStats::TSettings& statsSettings)
    : TRequestProcessor(NCrypta::NLog::GetLog("ping"), TaggedSingleton<TStats, decltype(*this)>("ping", statsSettings))
    , ReplicaDatabase(replicaDatabase)
    , PingId(NIdSerializer::FromString(pingConfig.GetPingId()))
    , TvmClient(tvmClient)
{
}

void TPingProcessor::DoProcess(NHttp::TRequestReply& reply, const TClient& clientInfo) {
    TScopeTimer scopeTimer(Stats.Percentile, "timing.process");

    Stats.Count->Add("request.total.received");
    Stats.Count->Add("tvm_client." + clientInfo.GetName());

    NHttp::TRequest request;
    try {
        request = NPingRequestParser::Parse(TCgiParameters(reply.GetRequestCgi()));
    } catch (const yexception& e) {
        SendResponse(reply, HTTP_BAD_REQUEST, e.what());
        return;
    }

    Stats.Count->Add("client." + clientInfo.GetName() + "." + request.Subclient);

    const auto tvmStatus = TvmClient.GetStatus();
    if (tvmStatus == NTvmAuth::TClientStatus::Error) {
        SendResponse(reply, HTTP_INTERNAL_SERVER_ERROR, ::TStringBuilder() << "Bad TVM client status: " << tvmStatus.GetCode());
        return;
    }

    const auto& dbState = TDbStateLoader(Log, THashSet<TString>()).Load(ReplicaDatabase, {PingId}, false, false);

    const auto* match = dbState.GetMatches().Get(PingId);
    if (!match) {
        Stats.Count->Add("request.status.not_found");
        SendResponse(reply, HTTP_NOT_FOUND, ::TStringBuilder() << "Nothing found for id " << NIdSerializer::ToString(PingId));
    } else {
        Stats.Count->Add("request.status.found");

        if (match->Contains(PING_RESPONSE_ID)) {
            SendResponse(reply, HTTP_OK, "OK");
        } else {
            SendResponse(reply, HTTP_INTERNAL_SERVER_ERROR,
                         ::TStringBuilder() << "Unexpected actual match for " << NIdSerializer::ToString(PingId) << ":\n"
                         << NMatchSerializer::ToJson(*match));
        }
    }
}
