#include "touch_processor.h"

#include <crypta/cm/services/api/lib/logic/touch/request/touch_request_parser.h>
#include <crypta/cm/services/common/data/id_utils.h>
#include <crypta/cm/services/common/db_state/get_changes_batch.h>
#include <crypta/cm/services/common/serializers/id/string/id_string_serializer.h>
#include <crypta/cm/services/common/serializers/match/record/match_record_serializer.h>
#include <crypta/cm/services/common/serializers/reply/json/reply_json_serializer.h>
#include <crypta/lib/native/singleton/tagged_singleton.h>
#include <crypta/lib/native/time/scope_timer.h>

#include <library/cpp/iterator/zip.h>

#include <util/string/builder.h>

using namespace NCrypta::NCm;
using namespace NCrypta::NCm::NApi;

TTouchProcessor::TTouchProcessor(
    NYtDynTables::TKvDatabase& replicaDatabase,
    TMutationSender& mutationSender,
    const TStats::TSettings& statsSettings,
    const TToucher& toucher
)
: TRequestProcessor(NCrypta::NLog::GetLog("touch"), TaggedSingleton<TStats, decltype(*this)>("touch", statsSettings))
    , ReplicaDatabase(replicaDatabase)
    , MutationSender(mutationSender)
    , Toucher(toucher)
{
}

void TTouchProcessor::DoProcess(NHttp::TRequestReply& reply, const TClient& clientInfo) {
    TScopeTimer scopeTimer(Stats.Percentile, "timing.process");

    Stats.Count->Add("request.total.received");
    Stats.Count->Add("tvm_client." + clientInfo.GetName());

    TTouchRequest request;
    try {
        request = NTouchRequestParser::Parse(TCgiParameters(reply.GetRequestCgi()), reply.GetRequestBody());
    } catch (const yexception& e) {
        SendResponse(reply, HTTP_BAD_REQUEST, e.what());
        return;
    }

    Stats.Count->Add("client." + clientInfo.GetName() + "." + request.Subclient);

    for (const auto& id: request.Ids) {
        if (IsInternalId(id)) {
            SendResponse(reply, HTTP_BAD_REQUEST, ::TStringBuilder() << "Can't touch internal id: " << NIdSerializer::ToString(id));
            return;
        }
    }

    const THashSet<TString> trackedBackRefTags;
    const auto dbState = TDbStateLoader(Log, trackedBackRefTags).Load(ReplicaDatabase, request.Ids, false, false);

    for (const auto& [id, touch_ts]: Zip(request.Ids, request.TouchTimestamps)) {
        const auto* match = dbState.GetMatches().Get(id);
        if (!match || match->GetInternalIds().empty()) {
            continue;
        }

        if (Toucher.NeedTouch(match->GetTouch(), touch_ts)) {
            Stats.Count->Add("ttl.touch_count");
            MutationSender.TouchMatch(*match, touch_ts);
        }
    }

    SendResponse(reply, HTTP_OK, "OK");
}
