#include "upload_processor.h"

#include <crypta/cm/services/api/lib/logic/upload/request/upload_request_parser.h>
#include <crypta/cm/services/api/lib/logic/upload/request/upload_request_validator.h>
#include <crypta/cm/services/common/serializers/id/string/id_string_serializer.h>
#include <crypta/lib/native/singleton/tagged_singleton.h>
#include <crypta/lib/native/time/scope_timer.h>

using namespace NCrypta;
using namespace NCrypta::NCm::NApi;

TUploadProcessor::TUploadProcessor(
    TMutationSender& mutationSender,
    const TStats::TSettings& statsSettings,
    const TLogOnlyTypes& logOnlyTypes,
    const TToucher& toucher,
    TQuoterClient& quoterClient)
    : TRequestProcessor(NLog::GetLog("upload"), TaggedSingleton<TStats, decltype(*this)>("upload", statsSettings))
    , MutationSender(mutationSender)
    , LogOnlyTypes(logOnlyTypes)
    , Toucher(toucher)
    , QuoterClient(quoterClient)
{
}

void TUploadProcessor::DoProcess(NHttp::TRequestReply& reply, const TClient& clientInfo) {
    TScopeTimer scopeTimer(Stats.Percentile, "timing.process");

    Stats.Count->Add("request.total.received");

    TUploadRequest request;
    try {
        request = NUploadRequestParser::Parse(TCgiParameters(reply.GetRequestCgi()), reply.GetRequestBody());
        NUploadRequestValidator::Validate(request);
    } catch (const yexception& e) {
        SendResponse(reply, HTTP_BAD_REQUEST, e.what());
        return;
    }

    Stats.Count->Add("client." + clientInfo.GetName() + "." + request.Subclient);

    auto& incomingMatch = request.Match;
    const auto& extIdType = incomingMatch.GetExtId().Type;

    if (LogOnlyTypes.IsLogOnly(extIdType)) {
        SendResponse(reply, HTTP_OK, "Logged but not actually stored since external id type '" + extIdType + "' is marked as log only");
        return;
    } else if (const auto& quotaState = QuoterClient.GetQuotaState(); quotaState.GetIsFull()) {
        SendResponse(reply, HTTP_SERVICE_UNAVAILABLE, "Service has run out of quota: " + quotaState.GetDescription());
        return;
    }

    const auto requestStartTime = reply.GetStartTime();
    Toucher.Touch(incomingMatch, requestStartTime);
    incomingMatch.SetMatchTs(requestStartTime);
    MutationSender.UpdateMatch(incomingMatch);
    Stats.Count->Add("request.total.updated");

    SendResponse(reply, HTTP_OK, "Scheduled for writing");
}
