#include "quoter_client.h"

#include <crypta/cm/services/quoter/grpc/quoter_service.grpc.pb.h>
#include <crypta/lib/native/graphite/utils.h>
#include <crypta/lib/native/singleton/tagged_singleton.h>

#include <grpc++/channel.h>
#include <grpc++/client_context.h>
#include <grpc++/create_channel.h>
#include <grpc/grpc.h>

#include <util/string/builder.h>
#include <util/string/cast.h>

#include <chrono>

using namespace NCrypta;
using namespace NCrypta::NCm;

const TString METRIC_UPDATE_OK = MakeGraphiteMetric("update", "ok");
const TString METRIC_UPDATE_ERROR = MakeGraphiteMetric("update", "error");
const TString METRIC_QUOTA_STATE_FULL = MakeGraphiteMetric("quota_state", "full");
const TString METRIC_QUOTA_STATE_NOT_FULL = MakeGraphiteMetric("quota_state", "not_full");

TQuoterClient::TQuoterClient(const TQuoterConfig& config, const TStats::TSettings& statsSettings)
    : Log(NLog::GetLog("quoter"))
    , Stats(TaggedSingleton<TStats, decltype(*this)>("quoter_client", statsSettings))
    , HostPort(config.GetHostPort())
    , EnvironmentType(config.GetEnvironmentType())
    , TimeoutSec(config.GetTimeoutSec())
    , CacheUpdater(TDuration::Seconds(config.GetRequestIntervalSec()), [this](){ UpdateState(); }, "QuoterClient", true)
{
    if (config.GetEnabled()) {
        CacheUpdater.Start();
    }
}

NQuoter::TQuotaState TQuoterClient::GetQuotaState() {
    TReadGuardBase<TLightRWLock> guard(CacheLock);
    return CachedState;
}

void TQuoterClient::UpdateState() {
    Log->debug("Updating quota state");
    auto channel = grpc::CreateChannel(HostPort, grpc::InsecureChannelCredentials());
    auto stub = NQuoter::TQuoterService::NewStub(channel);

    NQuoter::TGetQuotaRequest request;
    request.SetEnvironmentType(EnvironmentType);

    grpc::ClientContext context;
    std::chrono::time_point deadline = std::chrono::system_clock::now() + std::chrono::seconds(TimeoutSec);
    context.set_deadline(deadline);

    NQuoter::TQuotaState response;

    auto status = stub->GetQuotaState(&context, request, &response);
    if (!status.ok()) {
        const auto errorCode = static_cast<int>(status.error_code());
        const auto& errorMessage = status.error_message();
        Log->error("Request failed, code={}, {}", errorCode, errorMessage);
        Stats.Count->Add(METRIC_UPDATE_ERROR);

        response.SetIsFull(false);
        response.SetDescription(TStringBuilder() << "Request to quoter failed, code=" << ToString(errorCode) << ", " << errorMessage);
    } else {
        Stats.Count->Add(METRIC_UPDATE_OK);
    }

    Stats.Count->Add(response.GetIsFull() ? METRIC_QUOTA_STATE_FULL : METRIC_QUOTA_STATE_NOT_FULL);

    TWriteGuardBase<TLightRWLock> guard(CacheLock);
    CachedState = response;
}
