#include "worker.h"

#include "request_processor_factory.h"

#include <crypta/lib/native/singleton/tagged_singleton.h>

#include <library/cpp/http/misc/parsed_request.h>

#include <util/string/builder.h>

using namespace NCrypta::NCm;
using namespace NCrypta::NCm::NApi;
using namespace NCrypta::NHttp;

namespace {
    const TString METRIC_ERRORS_UNEXPECTED = "errors.unexpected";
    const TClient RESTRICTED_CLIENT_INFO;
}

TWorker::TWorker(
        NYtDynTables::TKvDatabase& replicaDatabase,
        TMutationSender& mutationSender,
        const TStats::TSettings& statsSettings,
        const TLogOnlyTypes& logOnlyTypes,
        const TPingConfig& pingConfig,
        const TClientsConfig& clientsConfig,
        const TDuration touchTimeout,
        TQuoterClient& quoterClient,
        const TTurboDecryptor& turboDecryptor,
        const NTvmAuth::TTvmClient& tvmClient)
    : RequestProcessorFactory(
        replicaDatabase,
        mutationSender,
        statsSettings,
        logOnlyTypes,
        pingConfig,
        touchTimeout,
        quoterClient,
        turboDecryptor,
        tvmClient)
    , ClientsConfig(clientsConfig)
    , Stats(TaggedSingleton<TStats, decltype(*this)>("worker", statsSettings))
    , Log(NLog::GetLog("worker"))
{
}

void TWorker::ProcessRequest(TRequestReply& reply) {
    try {
        const auto& clientInfo = GetClientInfo(reply.GetClientServiceTicket());
        auto* processor = RequestProcessorFactory.GetProcessor(reply.GetRequestMethod(), reply.GetRequestService(), clientInfo);

        if (processor != nullptr) {
            processor->Process(reply, clientInfo);
        } else {
            ReplyError(reply, HTTP_NOT_IMPLEMENTED,
                       TStringBuilder() << "Request " << reply.GetRequestMethod() << " " << reply.GetRequestService() << " is not supported");
        }

        Y_ENSURE(reply.IsReplied(), "Request not replied");
    } catch (const NTvmAuth::TTvmException& e) {
        ReplyError(reply, HTTP_UNAUTHORIZED, TStringBuilder() << "Invalid TVM ticket: " << e.what());
        return;
    } catch (const TProcessorFactory::TNotAllowedException& e) {
        ReplyError(reply, HTTP_FORBIDDEN, TStringBuilder() << "Permission denied: " << e.what());
        return;
    } catch (const std::exception& e) {
        Stats.Count->Add(METRIC_ERRORS_UNEXPECTED);
        ReplyError(reply, HTTP_INTERNAL_SERVER_ERROR, TStringBuilder() <<"Exception: " << e.what());
    } catch (...) {
        Stats.Count->Add(METRIC_ERRORS_UNEXPECTED);
        ReplyError(reply, HTTP_INTERNAL_SERVER_ERROR, "Unknown exception");
    }
}

void TWorker::ReplyError(TRequestReply& reply, HttpCodes httpCode, const TString& message) {
    reply.ReplyError(httpCode, message, Stats, Log);
}

const TClient& TWorker::GetClientInfo(const TMaybe<NTvmAuth::TCheckedServiceTicket>& clientServiceTicket) {
    if (!clientServiceTicket.Defined()) {
        Stats.Count->Add("request.tvm.no_ticket");
        return ClientsConfig.GetAnonymous();
    } else {
        const auto& tvmId = clientServiceTicket->GetSrc();
        const auto& clients = ClientsConfig.GetClients();

        const auto iter = clients.find(tvmId);
        if (iter == clients.end()) {
            Log->error("Unknown client TVM ID = {}", tvmId);
            Stats.Count->Add("request.tvm.unknown_client_id." + ToString(tvmId));
            return RESTRICTED_CLIENT_INFO;
        } else {
            return iter->second;
        }
    }
}
