#include "calc_expire_job.h"

#include "schemas.h"

#include <crypta/cm/services/calc_expire/lib/calc_expire_mapper.h>
#include <crypta/cm/services/calc_expire/lib/schemas.h>
#include <crypta/lib/native/time/shifted_clock.h>

#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/timed_yt_path_generator.h>

#include <google/protobuf/json_util.h>
#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <util/stream/str.h>

using namespace NCrypta::NCm::NCalcExpire;

TCalcExpireJob::TCalcExpireJob(const TCalcExpireJob::TConfig& config, NLog::TLogPtr log)
    : Config(config)
    , Log(log)
{
    Y_ENSURE(log != nullptr);
}

int TCalcExpireJob::Do() {
    Log->info("================ Start ================");
    TStringStream configJson;
    configJson << Config.AsJSON();
    Log->info("Config = {}", configJson.Str());

    NYT::TConfig::Get()->Pool = Config.GetYtPool();
    auto client = NYT::CreateClient(Config.GetYtProxy());

    FilterRowsToExpire(client);

    Log->info("================ Finish ================");
    return 0;
}

std::pair<NYT::TYPath, NYT::TYPath> TCalcExpireJob::FilterRowsToExpire(NYT::IClientPtr client) {
    const auto now = TShiftedClock::Now();
    const auto& dstPath = TTimedYtPathGenerator(now).GetPath(Config.GetDstDir());
    const auto& errorsPath = TTimedYtPathGenerator(now).GetPath(Config.GetErrorsDir());
    auto outputBuilder = TCalcExpireMapper::PrepareOutput(dstPath, errorsPath);

    auto tx = client->StartTransaction();

    auto spec = NYT::TMapOperationSpec();
    spec.AddInput<NYT::TNode>(Config.GetSrcTable());
    AddOutputs<NYT::TNode>(spec, outputBuilder.GetTables());

    const auto& specNode = NYT::TNode()
                               ("scheduling_tag_filter", Config.GetSchedulingTagFilter())
                               ("resource_limits", NYT::TNode()("user_slots", Config.GetMaxConcurrentJobs()));

    tx->Map(spec, new TCalcExpireMapper(outputBuilder.GetIndexes(), now), NYT::TOperationOptions().Spec(specNode));

    tx->Commit();

    SetTtl(client, errorsPath, TDuration::Days(Config.GetErrorsTtlDays()), ESetTtlMode::RemoveIfEmpty);

    return {dstPath, errorsPath};
}
