#include "calc_expire_mapper.h"

#include "fields.h"
#include "node_match_deserializer.h"
#include "schemas.h"

#include <crypta/cm/services/common/expiration/is_expired.h>
#include <crypta/cm/services/common/changes/change_commands.h>
#include <crypta/cm/services/common/data/id_utils.h>
#include <crypta/cm/services/common/serializers/id/string/id_string_serializer.h>
#include <crypta/lib/native/yexception_utils/yexception_utils.h>
#include <crypta/lib/native/yt/dyntables/kv_schema/fields.h>

using namespace NCrypta::NCm::NCalcExpire;
using namespace NCrypta::NYtDynTables;

const TString SUBKEY = "0";

TCalcExpireMapper::TCalcExpireMapper(const NCalcExpireMapper::TOutputIndexes& outputIndexes, TInstant startTs)
    : OutputIndexes(outputIndexes)
    , StartTs(startTs)
{
}

void TCalcExpireMapper::Do(TReader* reader, TWriter* writer) {
    for (; reader->IsValid(); reader->Next()) {
        const auto& row = reader->GetRow();
        const auto& key = row.At(NFields::KEY);
        try {
            const auto& id = NIdSerializer::FromString(key.AsString());
            if (IsInternalId(id)) {
                continue;
            }

            auto match = NNodeMatchDeserializer::DeserializeMatch(row);
            if (IsExpired(match, StartTs)) {
                const auto node = NYT::TNode()(NExpirationFields::TYPE, id.Type)
                                              (NExpirationFields::VALUE, id.Value)
                                              (NExpirationFields::UNIXTIME, StartTs.Seconds());
                writer->AddRow(node, OutputIndexes[NCalcExpireMapper::EOutputTables::ToExpire].GetRef());
            }
        }
        catch (const yexception& exc) {
            const auto node = NYT::TNode()(NExpirationFields::KEY, key)
                                          (NExpirationFields::VALUE, row.At(NFields::VALUE))
                                          (NExpirationFields::ERROR, RemovePathFromYexceptionMessage(exc.what()))
                                          (NExpirationFields::UNIXTIME, StartTs.Seconds());
            writer->AddRow(node, OutputIndexes[NCalcExpireMapper::EOutputTables::Errors].GetRef());
        }
    }
}

NCalcExpireMapper::TOutputIndexes::TBuilder TCalcExpireMapper::PrepareOutput(const NYT::TYPath& dst, const NYT::TYPath& errors) {
    NCalcExpireMapper::TOutputIndexes::TBuilder outputBuilder;

    outputBuilder.Add(NYT::TRichYPath(dst).Schema(CalcExpireSchema()).OptimizeFor(NYT::OF_SCAN_ATTR),
                      NCalcExpireMapper::EOutputTables::ToExpire);
    outputBuilder.Add(NYT::TRichYPath(errors).Schema(CalcExpireErrorsSchema()),
                      NCalcExpireMapper::EOutputTables::Errors);

    return outputBuilder;
}


REGISTER_MAPPER(TCalcExpireMapper);
