#include "ext_id_command.h"

#include "change_commands.h"
#include "change_log_fields.h"

#include <crypta/cm/services/common/data/id_validator.h>
#include <crypta/cm/services/common/serializers/id/string/id_string_serializer.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>

using namespace NCrypta::NCm;

TExtIdCommand::TExtIdCommand(const TString& command, const TString& shardingKey, const TId& extId, TInstant timestamp)
    : Command(command)
    , ShardingKey(shardingKey)
    , ExtId(extId)
    , Timestamp(timestamp)
{
}

bool TExtIdCommand::operator==(const NCrypta::NCm::TExtIdCommand& other) const {
    return (Command == other.Command)
        && (ShardingKey == other.ShardingKey)
        && (ExtId == other.ExtId)
        && (Timestamp == other.Timestamp);
}

bool TExtIdCommand::operator!=(const NCrypta::NCm::TExtIdCommand& other) const {
    return !(*this == other);
}

TString TExtIdCommand::ToString() const {
    TStringStream ss;
    NJson::TJsonWriter jsonWriter(&ss, false);
    jsonWriter.OpenMap();
    jsonWriter.Write(NChangeLogFields::CMD, Command);
    jsonWriter.Write(NChangeLogFields::EXT_ID, NIdSerializer::ToString(ExtId));
    jsonWriter.Write(NChangeLogFields::UNIXTIME, Timestamp.Seconds());
    jsonWriter.Write(NChangeLogFields::SHARDING_KEY, ShardingKey);
    jsonWriter.CloseMap();
    jsonWriter.Flush();
    return ss.Str();
 }

TExtIdCommand TExtIdCommand::FromJsonValue(const NJson::TJsonValue& v, const TString& type) {
    Y_ENSURE(v.IsMap(), "Command json must be a map");
    Y_ENSURE(v[NChangeLogFields::CMD].IsString(), "Field " << NChangeLogFields::CMD << " must be a string");
    Y_ENSURE(v[NChangeLogFields::EXT_ID].IsString(), "Field " << NChangeLogFields::EXT_ID << " must be a string");
    Y_ENSURE(v[NChangeLogFields::UNIXTIME].IsInteger(), "Field " << NChangeLogFields::UNIXTIME << " must be an integer");
    Y_ENSURE(v[NChangeLogFields::SHARDING_KEY].IsString(), "Field " << NChangeLogFields::SHARDING_KEY << " must be a string");

    Y_ENSURE(v[NChangeLogFields::CMD].GetString() == type, "Field '" << NChangeLogFields::CMD << "' must be equal to '" << type << "'");

    auto ret = TExtIdCommand(
        v[NChangeLogFields::CMD].GetString(),
        v[NChangeLogFields::SHARDING_KEY].GetString(),
        NIdSerializer::FromString(v[NChangeLogFields::EXT_ID].GetString()),
        TInstant::Seconds(v[NChangeLogFields::UNIXTIME].GetInteger())
    );
    Y_ENSURE(NIdValidator::IsValid(ret.ExtId), "Command must contain a valid ext_id");

    return ret;
}
