#include "touch_command.h"

#include "change_commands.h"
#include "change_log_fields.h"

#include <crypta/cm/services/common/data/id_validator.h>
#include <crypta/cm/services/common/serializers/id/string/id_string_serializer.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>

using namespace NCrypta::NCm;

TTouchCommand::TTouchCommand(const TString& shardingKey, const TId& extId, TInstant touchTs, TInstant timestamp)
    : ShardingKey(shardingKey)
    , ExtId(extId)
    , TouchTs(touchTs)
    , Timestamp(timestamp)
{
}

bool TTouchCommand::operator==(const NCrypta::NCm::TTouchCommand& other) const {
    return (ExtId == other.ExtId)
        && (TouchTs == other.TouchTs)
        && (Timestamp == other.Timestamp);
}

bool TTouchCommand::operator!=(const NCrypta::NCm::TTouchCommand& other) const {
    return !(*this == other);
}

TString TTouchCommand::ToString(const TTouchCommand& touchCommand) {
    TStringStream ss;
    NJson::TJsonWriter jsonWriter(&ss, false);
    jsonWriter.OpenMap();
    jsonWriter.Write(NChangeLogFields::CMD, NChangeCommands::TOUCH);
    jsonWriter.Write(NChangeLogFields::EXT_ID, NIdSerializer::ToString(touchCommand.ExtId));
    jsonWriter.Write(NChangeLogFields::TOUCH_TS, touchCommand.TouchTs.Seconds());
    jsonWriter.Write(NChangeLogFields::UNIXTIME, touchCommand.Timestamp.Seconds());
    jsonWriter.Write(NChangeLogFields::SHARDING_KEY, touchCommand.ShardingKey);
    jsonWriter.CloseMap();
    jsonWriter.Flush();
    return ss.Str();

}

TTouchCommand TTouchCommand::FromJsonValue(const NJson::TJsonValue& v) {
    Y_ENSURE(v.IsMap(), "Command json must be a map");
    Y_ENSURE(v[NChangeLogFields::CMD].IsString(), "Field " << NChangeLogFields::CMD << " must be a string");
    Y_ENSURE(v[NChangeLogFields::EXT_ID].IsString(), "Field " << NChangeLogFields::EXT_ID << " must be a string");
    Y_ENSURE(v[NChangeLogFields::TOUCH_TS].IsInteger(), "Field " << NChangeLogFields::TOUCH_TS << " must be an integer");
    Y_ENSURE(v[NChangeLogFields::UNIXTIME].IsInteger(), "Field " << NChangeLogFields::UNIXTIME << " must be an integer");
    Y_ENSURE(v[NChangeLogFields::SHARDING_KEY].IsString(), "Field " << NChangeLogFields::SHARDING_KEY << " must be a string");

    Y_ENSURE(v[NChangeLogFields::CMD].GetString() == NChangeCommands::TOUCH, "Field '" << NChangeLogFields::CMD << "' must be equal to '" << NChangeCommands::TOUCH << "'");

    auto ret = TTouchCommand(
        v[NChangeLogFields::SHARDING_KEY].GetString(),
        NIdSerializer::FromString(v[NChangeLogFields::EXT_ID].GetString()),
        TInstant::Seconds(v[NChangeLogFields::TOUCH_TS].GetInteger()),
        TInstant::Seconds(v[NChangeLogFields::UNIXTIME].GetInteger())
    );
    Y_ENSURE(NIdValidator::IsValid(ret.ExtId), "Command must contain a valid ext_id");

    return ret;
}
