#include "upload_command.h"

#include "change_commands.h"
#include "change_log_fields.h"

#include <crypta/cm/services/common/data/match_validator.h>
#include <crypta/cm/services/common/serializers/match/proto/match_proto_serializer.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>

#include <library/cpp/string_utils/base64/base64.h>

using namespace NCrypta::NCm;

namespace {
    TString SerializeMatch(const TMatch& match) {
        return Base64Encode(NMatchSerializer::ToProtoString(match));
    }

    TMatch DeserializeMatch(const TString& str) {
        return NMatchSerializer::FromProtoString(Base64Decode(str));
    }
}

TUploadCommand::TUploadCommand(const TString& shardingKey, const TMatch& incomingMatch, TInstant timestamp)
    : ShardingKey(shardingKey)
    , IncomingMatch(incomingMatch)
    , Timestamp(timestamp)
{
}

bool TUploadCommand::operator==(const TUploadCommand& other) const {
    return IncomingMatch == other.IncomingMatch;
}

bool TUploadCommand::operator!=(const TUploadCommand& other) const {
    return !(*this == other);
}

TString TUploadCommand::ToString(const TUploadCommand& uploadCommand) {
    TStringStream ss;
    NJson::TJsonWriter jsonWriter(&ss, false);
    jsonWriter.OpenMap();
    jsonWriter.Write(NChangeLogFields::CMD, NChangeCommands::UPLOAD);
    jsonWriter.Write(NChangeLogFields::MATCH, SerializeMatch(uploadCommand.IncomingMatch));
    jsonWriter.Write(NChangeLogFields::UNIXTIME, uploadCommand.Timestamp.Seconds());
    jsonWriter.Write(NChangeLogFields::SHARDING_KEY, uploadCommand.ShardingKey);
    jsonWriter.CloseMap();
    jsonWriter.Flush();
    return ss.Str();
}

TUploadCommand TUploadCommand::FromJsonValue(const NJson::TJsonValue& v) {
    Y_ENSURE(v.IsMap(), "Command json must be a map");
    Y_ENSURE(v[NChangeLogFields::CMD].IsString(), "Field " << NChangeLogFields::CMD << " must be a string");
    Y_ENSURE(v[NChangeLogFields::MATCH].IsString(), "Field " << NChangeLogFields::MATCH << " must be a string");
    Y_ENSURE(v[NChangeLogFields::UNIXTIME].IsInteger(), "Field " << NChangeLogFields::UNIXTIME << " must be an integer");
    Y_ENSURE(v[NChangeLogFields::SHARDING_KEY].IsString(), "Field " << NChangeLogFields::SHARDING_KEY << " must be a string");

    Y_ENSURE(v[NChangeLogFields::CMD].GetString() == NChangeCommands::UPLOAD, "Field '" << NChangeLogFields::CMD << "' must be equal to '" << NChangeCommands::UPLOAD << "'");

    auto ret = TUploadCommand(
        v[NChangeLogFields::SHARDING_KEY].GetString(),
        DeserializeMatch(v[NChangeLogFields::MATCH].GetString()),
        TInstant::Seconds(v[NChangeLogFields::UNIXTIME].GetInteger())
    );
    Y_ENSURE(NMatchValidator::IsValid(ret.IncomingMatch), "Command must contain a valid match");

    return ret;
}
