#pragma once

#include <crypta/lib/native/database/record.h>
#include <crypta/cm/services/common/data/id.h>
#include <crypta/cm/services/common/data/id_hash.h>

#include <util/generic/hash.h>
#include <util/generic/ptr.h>
#include <util/generic/vector.h>

namespace NCrypta::NCm {
    template <typename T>
    class TChangeTrackingMap {
    private:
        enum class EStatus {
            Updated,
            Deleted
        };
        using TStatusMap = THashMap<TId, EStatus>;

    public:
        using TMap = THashMap<TId, T>;

        class TIterator {
        public:
            using value_type = const TId;
            using reference = const TId&;
            using pointer = const TId*;
            using difference_type = i64;
            using iterator_category = std::forward_iterator_tag;

            TIterator(const TChangeTrackingMap<T>& map, EStatus status, bool end)
                : Iter(end ? map.Statuses.cend() : map.Statuses.cbegin())
                , End(map.Statuses.cend())
                , Status(status)
            {
                FindNext();
            }

            reference operator*() const {
                return Iter->first;
            }

            TIterator& operator++() {
                ++Iter;
                FindNext();
                return *this;
            }

            bool operator==(const TIterator& other) {
                return Iter == other.Iter;
            }

            bool operator!=(const TIterator& other) {
                return !(*this == other);
            }

        private:
            void FindNext() {
                while (Iter != End && Iter->second != Status) {
                    ++Iter;
                }
            }

            typename TStatusMap::const_iterator Iter;
            const typename TStatusMap::const_iterator End;
            const EStatus Status;
        };

        class TUpdateIterator : public TIterator {
        public:
            using value_type = const T;
            using reference = const T&;
            using pointer = const T*;

            TUpdateIterator(const TChangeTrackingMap<T>& map, bool end)
                : TIterator(map, EStatus::Updated, end)
                , Map(map.Map) {
            }

            reference operator*() const {
                return Map.at(TIterator::operator*());
            }

            TUpdateIterator& operator++() {
                TIterator::operator++();
                return *this;
            }

        private:
            const TMap& Map;
        };

        class TDeleteIterator : public TIterator {
        public:
            TDeleteIterator(const TChangeTrackingMap<T>& map, bool end)
                : TIterator(map, EStatus::Deleted, end) {
            }

            TDeleteIterator& operator++() {
                TIterator::operator++();
                return *this;
            }
        };

        TChangeTrackingMap() = default;

        TChangeTrackingMap(TMap map)
            : Map(std::move(map)) {
        }

        const T* Get(const TId& key) const {
            const auto& it = Map.find(key);
            return it == Map.end() ? nullptr : &it->second;
        }

        void Delete(const TId& key) {
            Y_ENSURE(Map.erase(key), "Tried to delete unknown key");
            Statuses[key] = EStatus::Deleted;
        }

        void Write(const TId& key, T value) {
            Map[key] = std::move(value);
            Statuses[key] = EStatus::Updated;
        }

        void Update(const TId& key, const std::function<bool(T&)>& updater) {
            if (updater(Map.at(key))) {
                Statuses[key] = EStatus::Updated;
            }
        };

        TUpdateIterator UpdateBegin() const {
            return TUpdateIterator(*this, false);
        }

        TUpdateIterator UpdateEnd() const {
            return TUpdateIterator(*this, true);
        }

        TDeleteIterator DeleteBegin() const {
            return TDeleteIterator(*this, false);
        }

        TDeleteIterator DeleteEnd() const {
            return TDeleteIterator(*this, true);
        }

    private:
        TStatusMap Statuses;
        TMap Map;
    };
}
