#include "db_state.h"

#include <crypta/cm/services/common/data/matched_id_comparator.h>
#include <crypta/cm/services/common/serializers/back_reference/record/back_reference_record_serializer.h>
#include <crypta/cm/services/common/serializers/id/string/id_string_serializer.h>

using namespace NCrypta::NCm;

namespace {
    TMatch Merge(const TMatch& incomingMatch, const TMatch& currentMatch) {
        TMatch::TMatchedIds matchedIds = currentMatch.GetInternalIds();

        for (const auto& [type, incomingMatchedId] : incomingMatch.GetInternalIds()) {
            const auto* currentMatchedId = currentMatch.FindInternalId(type);
            if (currentMatchedId == nullptr || (
               !(incomingMatchedId.IsRealtime() && !currentMatchedId->IsRealtime()) &&
               !NMatchedIdComparator::Equal(*currentMatchedId, incomingMatchedId, NMatchedIdComparator::EMode::Attributes | NMatchedIdComparator::EMode::Id)
            )) {
                matchedIds[type] = incomingMatchedId;
            }
        }

        return TMatch(
            incomingMatch.GetExtId(),
            std::move(matchedIds),
            incomingMatch.GetTouch(),
            incomingMatch.GetTtl(),
            incomingMatch.GetTrackBackReference()
        );
    }

    bool UpdateBackRefWithLimit(TBackReference& backRef, const TId& newExtId) {
        THashSet<TId> toDelete;
        bool hasNewExtId = false;
        for (auto& oldExtId : backRef.Refs) {
            if (oldExtId.Type == newExtId.Type) {
                if (oldExtId != newExtId) {
                    toDelete.emplace(oldExtId);
                } else {
                    hasNewExtId = true;
                }
            }
        }

        if (toDelete.empty() && hasNewExtId) {
            return false;
        }

        for (const auto& extId : toDelete) {
            backRef.Refs.erase(extId);
        }
        if (!hasNewExtId) {
            backRef.Refs.emplace(newExtId);
        }

        return true;
    }
}


TDbState::TDbState(TMatchMap::TMap matchesMap, TBackRefMap::TMap backRefsMap, const THashSet<TString>& trackedBackRefTags)
    : Matches(std::move(matchesMap))
    , BackRefs(std::move(backRefsMap))
    , TrackedBackRefTags(trackedBackRefTags)
{
}

void TDbState::WriteMatch(const TMatch& incomingMatch) {
    const auto* dbMatch = Matches.Get(incomingMatch.GetExtId());
    const auto& finalMatch = dbMatch ? Merge(incomingMatch, *dbMatch) : incomingMatch;

    if (finalMatch.GetTrackBackReference() || TrackedBackRefTags.contains(incomingMatch.GetExtId().Type)) {
        for (const auto& [type, matchedId] : finalMatch.GetInternalIds()) {
            if (dbMatch) {
                RebindBackRef(matchedId.GetId(), *dbMatch);
            } else {
                AddBackRef(matchedId.GetId(), finalMatch.GetExtId());
            }
        }
    }

    Matches.Write(
        finalMatch.GetExtId(),
        finalMatch
    );
}

void TDbState::DeleteMatch(const TId& extId) {
    const auto* match = Matches.Get(extId);

    if (!match) {
        return;
    }

    if (match->GetTrackBackReference() || TrackedBackRefTags.contains(extId.Type)) {
        for (const auto& [type, matchedId] : match->GetInternalIds()) {
            RemoveBackRef(matchedId.GetId(), extId);
        }
    }

    Matches.Delete(extId);
}

void TDbState::TouchMatch(const TId& extId, const TInstant touchTs, const TDuration ttl) {
    Matches.Update(extId, [=](TMatch& match) {
        match.SetTouch(touchTs);
        match.SetTtl(ttl);
        return true;
    });
}

const TDbState::TMatchMap& TDbState::GetMatches() const {
    return Matches;
}

const TDbState::TBackRefMap& TDbState::GetBackRefs() const {
    return BackRefs;
};

void TDbState::RebindBackRef(const TId& newMatchedId, const TMatch& oldMatch) {
    AddBackRef(newMatchedId, oldMatch.GetExtId());

    auto internalId = oldMatch.FindInternalId(newMatchedId.Type);
    if (internalId == nullptr) {
        return;
    }

    const auto& oldId = internalId->GetId();
    if (oldId == newMatchedId) {
        return;
    }

    RemoveBackRef(oldId, oldMatch.GetExtId());
}

void TDbState::AddBackRef(const TId& newMatchedId, const TId& extId) {
    if (BackRefs.Get(newMatchedId)) {
        BackRefs.Update(newMatchedId, [&extId](TBackReference& backRef){
            return UpdateBackRefWithLimit(backRef, extId);
        });
    } else {
        BackRefs.Write(newMatchedId, TBackReference(newMatchedId, {extId}));
    }
}

void TDbState::RemoveBackRef(const TId& matchedId, const TId& extId) {
    const auto* backRef = BackRefs.Get(matchedId);
    if (!backRef) {
        return;
    }

    Y_ENSURE(!backRef->Refs.empty(), "Empty back reference");
    const auto& it = backRef->Refs.find(extId);
    if (it == backRef->Refs.end()) {
        return;
    }

    if (backRef->Refs.size() == 1) {
        BackRefs.Delete(matchedId);
    } else {
        BackRefs.Update(matchedId, [&](auto& backRef) -> bool {
            backRef.Refs.erase(it);
            return true;
        });
    }
}
