#include "db_state_loader.h"

#include <crypta/cm/services/common/data/id_utils.h>
#include <crypta/cm/services/common/serializers/back_reference/record/back_reference_record_serializer.h>
#include <crypta/cm/services/common/serializers/id/string/id_string_serializer.h>
#include <crypta/cm/services/common/serializers/match/record/match_record_serializer.h>

using namespace NCrypta::NCm;

TDbStateLoader::TDbStateLoader(NLog::TLogPtr& log, const THashSet<TString>& trackedBackRefTags)
    : Log(log)
    , TrackedBackRefTags(trackedBackRefTags)
{
}

TDbState TDbStateLoader::Load(TDbStateLoader::TDatabase& database, const TVector<TId>& ids, bool loadSecondOrderBackRefs, bool loadSecondOrderMatches) {
    LookupBatch.clear();
    Matches.clear();
    BackRefs.clear();

    EnqueueIds(ids);

    LoadOneStep(database, loadSecondOrderBackRefs, loadSecondOrderMatches);

    if (!LookupBatch.empty()) {
        LoadOneStep(database, loadSecondOrderBackRefs, loadSecondOrderMatches);
    }

    return TDbState(std::move(Matches), std::move(BackRefs), TrackedBackRefTags);
}

void TDbStateLoader::LoadOneStep(TDatabase& database, bool loadSecondOrderBackRefs, bool loadSecondOrderMatches) {
    const auto records = database.Lookup(LookupBatch);

    TVector<TId> idsForNextStep;
    for (const auto& record : records) {
        const auto it = RequestedIds.find(record.Key);
        Y_ENSURE(it != RequestedIds.end(), "Loaded id was not requested");

        Log->info("Found id = {} from KV", record.Key);

        RequestedIds.erase(it);

        if (IsInternalId(NIdSerializer::FromString(record.Key))) {
            auto backReference = NBackReferenceSerializer::FromRecord(record);

            if (loadSecondOrderMatches) {
                for (const auto& extId : backReference.Refs) {
                    idsForNextStep.push_back(extId);
                }
            }

            BackRefs.emplace(TId(backReference.Id), std::move(backReference));
        } else {
            auto match = NMatchSerializer::FromRecord(record);

            if (loadSecondOrderBackRefs && (match.GetTrackBackReference() ||TrackedBackRefTags.contains(match.GetExtId().Type))) {
                for (const auto&[type, matchedId] : match.GetInternalIds()) {
                    idsForNextStep.push_back(matchedId.GetId());
                }
            }

            Matches.emplace(TId(match.GetExtId()), std::move(match));
        }
    }

    for (const auto& id : RequestedIds) {
        Log->info("Not found id = {} in KV", id);
    }

    RequestedIds.clear();

    EnqueueIds(idsForNextStep);
}

void TDbStateLoader::EnqueueIds(const TVector<TId>& ids) {
    LookupBatch.clear();
    LookupBatch.reserve(ids.size());

    for (const auto& id : ids) {
        auto[it, actuallyInserted] = SeenIds.insert(id);

        if (actuallyInserted) {
            const auto& idStr = NIdSerializer::ToString(*it);
            LookupBatch.push_back(idStr);
            RequestedIds.insert(idStr);
        }
    }
}
