import collections
import time
import urllib

from crypta.cm.services.common.ext_id_mapper.lib import errors
from crypta.lib.python.yt.http_mapper import tvm_http_requests_mapper


TYPE = "type"
VALUE = "value"


def get_url_template(handle, subclient):
    return "/{}?subclient={}".format(handle, subclient) + "&type={type}&value={value}"


class ExtIdMapper(tvm_http_requests_mapper.TvmHttpRequestsMapper):
    def __init__(self, max_retries, *args, **kwargs):
        super(ExtIdMapper, self).__init__(*args, **kwargs)

        self.retry_counter = collections.Counter()
        self.max_retries = max_retries
        self.batch_size = 1

    def get_request_wo_headers(self, batch):
        row = batch[0]
        yield "GET", self.url.format(type=urllib.quote(row[TYPE]), value=urllib.quote(row[VALUE])), None

    def is_request_retriable(self, request):
        return (self.retry_counter[self._get_counter_key(request.batch[0])] < self.max_retries) and (request.status != "404")

    def retry_failed_request(self, request):
        self.retry_counter.update([self._get_counter_key(request.batch[0])])
        return self.is_request_retriable(request)

    def process_response(self, request):
        row = request.batch[0]
        counter_key = self._get_counter_key(row)
        request_succeded = request.ok or request.status == "404"

        if not request_succeded and not self.is_request_retriable(request):
            yield {
                errors.TYPE: row[TYPE],
                errors.VALUE: row[VALUE],
                errors.STATUS: request.status,
                errors.DATA: request.response_body,
                errors.UNIXTIME: int(time.time()),
            }
        elif request_succeded and counter_key in self.retry_counter:
            del self.retry_counter[counter_key]

    def _get_counter_key(self, row):
        return (row[TYPE], row[VALUE])
