#include "match_json_serializer.h"

#include <crypta/cm/services/common/data/match_validator.h>
#include <crypta/cm/services/common/serializers/matched_ids/json/matched_ids_json_serializer.h>
#include <crypta/cm/services/common/serializers/matched_ids/json/matched_ids_serializer_utils.h>

#include <library/cpp/json/json_writer.h>
#include <library/cpp/json/json_reader.h>

#include <util/generic/hash_set.h>

using namespace NCrypta::NCm;
using namespace NJson;

namespace {
    const bool FORMAT_OUTPUT = false;
    const bool SORT_KEYS = false;
}

TString NMatchSerializer::ToJson(const TMatch& match) {
    // The optimal way:
    // TJsonWriter json(&out, FORMAT_OUTPUT);
    // https://a.yandex-team.ru/arc/trunk/arcadia/library/cpp/json/ut/json_writer_ut.cpp?rev=4085656#L16

    TJsonValue v;
    v["ext_id"]["type"] = match.GetExtId().Type;
    v["ext_id"]["value"] = match.GetExtId().Value;

    v["ids"] = NMatchedIdsJsonSerializer::ToJsonValue(match.GetInternalIds());

    if (match.GetTrackBackReference()) {
        v["track_back_reference"] = true;
    }

    TStringStream out;
    WriteJson(&out, &v, FORMAT_OUTPUT, SORT_KEYS);
    return out.Str();
}

TMatch NMatchSerializer::FromJson(const TStringBuf& str) {
    //TODO(r-andrey): The optimal way to deserialize is via json callbacks.

    TJsonValue v;
    ReadJsonTree(str, &v, true);

    if (!v.IsMap()) {
        ythrow yexception() << "Json must be a map";
    }

    TMatch match;
    match.SetExtId({v["ext_id"]["type"].GetString(), v["ext_id"]["value"].GetString()});

    const auto& serializedIds = v["ids"].GetArray();
    for (const auto& id: serializedIds) {
        TAttributes attrs;
        DeserializeAttributes(id["attributes"].GetMap(), attrs);
        match.AddId(TMatchedId(
            TId(id["type"].GetString(), id["value"].GetString()),
            TInstant::Seconds(id["match_ts"].GetUInteger()),
            id["cas"].GetUInteger(),
            attrs
        ));
    }

    const auto& trackBackReference = v["track_back_reference"];
    if (trackBackReference.IsDefined()) {
        if (!trackBackReference.IsBoolean()) {
            ythrow yexception() << "track_back_reference must be boolean";
        }
        match.SetTrackBackReference(trackBackReference.GetBoolean());
    }

    Y_ENSURE(NMatchValidator::IsValid(match), "Ext id match is not valid");

    return match;
}
