#include "matched_ids_json_serializer.h"
#include "matched_ids_serializer_utils.h"

#include <library/cpp/json/json_writer.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/iterator/functools.h>

#include <util/stream/str.h>

using namespace NCrypta::NCm;
using namespace NJson;

namespace {
    const bool FORMAT_OUTPUT = false;
    const bool SORT_KEYS = false;

    template <typename TContainer>
    TJsonValue ToJsonValue(const TContainer& matchedIds) {
        TJsonValue ret(EJsonValueType::JSON_ARRAY);
        for (const auto& id : matchedIds) {
            TJsonValue jsonId(EJsonValueType::JSON_MAP);
            jsonId["type"] = id.GetId().Type;
            jsonId["value"] = id.GetId().Value;
            jsonId["match_ts"] = id.GetMatchTs().Seconds();
            jsonId["cas"] = id.GetCas();
            jsonId["attributes"] = TJsonValue(EJsonValueType::JSON_MAP);
            for (const auto& attr : id.GetAttributes()) {
                jsonId["attributes"][attr.first] = attr.second;
            }
            ret.AppendValue(jsonId);
        }
        return ret;
    }

    template <typename TContainer>
    TString Serialize(const TContainer& ids) {
        TStringStream out;
        const auto& v = NMatchedIdsJsonSerializer::ToJsonValue(ids);
        WriteJson(&out, &v, FORMAT_OUTPUT, SORT_KEYS);
        return out.Str();
    }
}

TString NMatchedIdsJsonSerializer::Serialize(const TMatch::TMatchedIds& ids) {
    return ::Serialize(ids);
}

TString NMatchedIdsJsonSerializer::Serialize(const TMatchedIds& ids) {
    return ::Serialize(ids);
}

TJsonValue NMatchedIdsJsonSerializer::ToJsonValue(const TMatch::TMatchedIds& matchedIds) {
    return ::ToJsonValue(NFuncTools::Map([](const auto& pair){ return pair.second; }, matchedIds));
}

TJsonValue NMatchedIdsJsonSerializer::ToJsonValue(const TMatchedIds& matchedIds) {
    return ::ToJsonValue(matchedIds);
}

TMatchedIds NMatchedIdsJsonSerializer::Deserialize(const TStringBuf& str) {
    TMatchedIds ids;
    Deserialize(str, ids);
    return ids;
}

void NMatchedIdsJsonSerializer::Deserialize(const TStringBuf& str, TMatchedIds& matchedIds) {
    //TODO(r-andrey): The optimal way to deserialize is via json callbacks.

    TJsonValue v;
    ReadJsonTree(str, &v);

    const auto& serializedIds = v.GetArray();
    matchedIds.reserve(serializedIds.size());
    for (const auto& id : serializedIds) {
        TAttributes attrs;
        DeserializeAttributes(id["attributes"].GetMap(), attrs);
        matchedIds.emplace_back(TMatchedId(
                TId(id["type"].GetString(), id["value"].GetString()),
                TInstant::Seconds(id["match_ts"].GetUInteger()),
                id["cas"].GetUInteger(),
                attrs
        ));
    }
}
